<?php

namespace Drupal\content_completeness_index\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\content_completeness_index\Service\CompletenessCalculator;
use Drupal\content_completeness_index\Service\CompletenessConfigManager;
use Drupal\content_completeness_index\Service\CompletenessIndexStorage;
use Drupal\content_completeness_index\Service\FieldGroupingHelper;
use Drupal\node\NodeInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a 'Content Completeness Index Floating Form' block.
 *
 * @Block(
 *   id = "content_completeness_index_floating_form",
 *   admin_label = @Translation("Content Completeness Index - Floating Form Assistant"),
 *   category = @Translation("Content Completeness Index"),
 *   context_definitions = {
 *     "node" = @ContextDefinition("entity:node", label = @Translation("Node"), required = FALSE)
 *   }
 * )
 */
class ContentCompletenessIndexFloatingFormBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * The completeness index storage service.
   *
   * @var \Drupal\content_completeness_index\Service\CompletenessIndexStorage
   */
  protected CompletenessIndexStorage $storage;

  /**
   * The config manager service.
   *
   * @var \Drupal\content_completeness_index\Service\CompletenessConfigManager
   */
  protected CompletenessConfigManager $configManager;

  /**
   * The calculator service.
   *
   * @var \Drupal\content_completeness_index\Service\CompletenessCalculator
   */
  protected CompletenessCalculator $calculator;

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected EntityFieldManagerInterface $entityFieldManager;

  /**
   * The route match service.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected RouteMatchInterface $routeMatch;

  /**
   * Helper service for deriving field grouping information.
   *
   * @var \Drupal\content_completeness_index\Service\FieldGroupingHelper
   */
  protected FieldGroupingHelper $fieldGroupingHelper;

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return parent::defaultConfiguration() + [
      'start_collapsed' => FALSE,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state) {
    $form = parent::blockForm($form, $form_state);
    $config = $this->getConfiguration();

    $form['start_collapsed'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Start collapsed'),
      '#description' => $this->t('Display the assistant in its collapsed state by default. Users can expand it manually.'),
      '#default_value' => !empty($config['start_collapsed']),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state) {
    parent::blockSubmit($form, $form_state);
    $this->configuration['start_collapsed'] = (bool) $form_state->getValue('start_collapsed');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = new static($configuration, $plugin_id, $plugin_definition);
    $instance->storage = $container->get('content_completeness_index.storage');
    $instance->configManager = $container->get('content_completeness_index.config_manager');
    $instance->calculator = $container->get('content_completeness_index.calculator');
    $instance->entityFieldManager = $container->get('entity_field.manager');
    $instance->routeMatch = $container->get('current_route_match');
    $instance->fieldGroupingHelper = $container->get('content_completeness_index.field_group_helper');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    $build = [];

    // Try to get node from context or current route.
    $node = $this->getContextValue('node');
    if (!$node instanceof NodeInterface) {
      $node = $this->routeMatch->getParameter('node');
    }

    if (!$node instanceof NodeInterface) {
      return $build;
    }

    $entity_type = $node->getEntityTypeId();
    $bundle = $node->bundle();
    $revision_id = $node->getRevisionId();

    // Check if completeness is enabled for this bundle.
    $config = $this->configManager->getConfig($bundle);
    if (!$config['enabled']) {
      return $build;
    }

    // Get current score.
    $score = $revision_id ? $this->storage->getScoreForRevision($entity_type, $revision_id) : 0;

    // Get configuration.
    $weights = $config['weights'] ?? [];

    // Calculate missing fields.
    $missing_fields = [];
    $field_definitions = $this->entityFieldManager->getFieldDefinitions($entity_type, $bundle);

    foreach ($weights as $field_name => $weight) {
      if ($weight > 0 && $node->hasField($field_name)) {
        $field = $node->get($field_name);
        if ($field->isEmpty()) {
          $field_definition = $field_definitions[$field_name] ?? NULL;
          if ($field_definition) {
            $missing_fields[] = [
              'name' => $field_name,
              'label' => $field_definition->getLabel(),
              'weight' => $weight,
              'anchor' => $this->buildFieldAnchor($field_name),
            ];
          }
        }
      }
    }

    $grouped_missing = $this->fieldGroupingHelper->groupFields($entity_type, $bundle, $missing_fields);
    $start_collapsed = !empty($this->configuration['start_collapsed']);

    $build = [
      '#type' => 'component',
      '#component' => 'content_completeness_index:content-completeness-index-floating-form',
      '#props' => [
        'score' => $score,
        'bundle' => $bundle,
        'entity_type' => $entity_type,
        'missing_groups' => $grouped_missing['groups'],
        'missing_fields' => $grouped_missing['ungrouped'],
        'start_collapsed' => $start_collapsed,
      ],
    ];

    return $build;
  }

  /**
   * Builds an in-page anchor for the given field name.
   */
  protected function buildFieldAnchor(string $field_name): string {
    return 'edit-' . strtr($field_name, ['_' => '-']) . '-wrapper';
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheContexts() {
    return ['url.path', 'route'];
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheTags() {
    $tags = parent::getCacheTags();

    // Add node cache tag if available.
    $node = $this->getContextValue('node');
    if (!$node instanceof NodeInterface) {
      $node = $this->routeMatch->getParameter('node');
    }

    if ($node instanceof NodeInterface) {
      $tags = array_merge($tags, $node->getCacheTags());
    }

    return $tags;
  }

}
