<?php

namespace Drupal\content_completeness_index\Service;

use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * Service for managing completeness index configuration.
 */
class CompletenessConfigManager {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * Constructs a CompletenessConfigManager object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(ConfigFactoryInterface $config_factory) {
    $this->configFactory = $config_factory;
  }

  /**
   * Gets the configuration for a specific bundle.
   *
   * @param string $bundle
   *   The bundle name.
   *
   * @return array
   *   The configuration array with keys: enabled, weights, total_weight.
   */
  public function getConfig(string $bundle): array {
    $config = $this->configFactory->get("content_completeness_index.settings.{$bundle}");

    $stored_weights = $config->get('weights') ?? [];
    $field_settings = $config->get('field_settings') ?? [];

    // Backwards compatibility: build field settings from legacy weight config.
    if (empty($field_settings) && !empty($stored_weights)) {
      foreach ($stored_weights as $field_name => $weight) {
        $field_settings[$field_name] = [
          'enabled' => $weight > 0,
          'weight' => $weight,
        ];
      }
    }

    $weights = [];
    if (!empty($field_settings)) {
      foreach ($field_settings as $field_name => $settings) {
        $weight = isset($settings['weight']) ? (float) $settings['weight'] : 0;
        if (!empty($settings['enabled'])) {
          $weights[$field_name] = $weight;
        }
      }
    }
    else {
      $weights = $stored_weights;
    }

    $total_weight = 0;
    foreach ($weights as $weight) {
      if ($weight > 0) {
        $total_weight += $weight;
      }
    }

    return [
      'enabled' => $config->get('enabled') ?? FALSE,
      'weights' => $weights,
      'field_settings' => $field_settings,
      'total_weight' => $total_weight,
      'wprogress_description' => $config->get('wprogress_description') ?? '',
    ];
  }

  /**
   * Saves the configuration for a specific bundle.
   *
   * @param string $bundle
   *   The bundle name.
   * @param array $config
   *   The configuration array.
   */
  public function saveConfig(string $bundle, array $config): void {
    $config_object = $this->configFactory->getEditable("content_completeness_index.settings.{$bundle}");

    $config_object->set('enabled', $config['enabled'] ?? FALSE);
    $config_object->set('weights', $config['weights'] ?? []);
    $config_object->set('field_settings', $config['field_settings'] ?? []);
    $config_object->set('total_weight', $config['total_weight'] ?? 0);
    $config_object->set('wprogress_description', $config['wprogress_description'] ?? '');

    $config_object->save();
  }

  /**
   * Checks if completeness scoring is enabled for a bundle.
   *
   * @param string $bundle
   *   The bundle name.
   *
   * @return bool
   *   TRUE if enabled, FALSE otherwise.
   */
  public function isEnabled(string $bundle): bool {
    return $this->configFactory->get("content_completeness_index.settings.{$bundle}")->get('enabled') ?? FALSE;
  }

  /**
   * Gets all bundles with completeness scoring enabled.
   *
   * @return array
   *   Array of bundle names.
   */
  public function getEnabledBundles(): array {
    $bundles = [];
    $config_names = $this->configFactory->listAll('content_completeness_index.settings.');

    foreach ($config_names as $config_name) {
      $config = $this->configFactory->get($config_name);
      if ($config->get('enabled')) {
        // Extract bundle name from config name.
        $bundle = str_replace('content_completeness_index.settings.', '', $config_name);
        $bundles[] = $bundle;
      }
    }

    return $bundles;
  }

}
