<?php

namespace Drupal\content_completeness_index\Commands;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\content_completeness_index\Service\CompletenessCalculator;
use Drupal\content_completeness_index\Service\CompletenessConfigManager;
use Drupal\content_completeness_index\Service\CompletenessIndexStorage;
use Drush\Attributes\Command;
use Drush\Attributes\Argument;
use Drush\Attributes\Option;
use Drush\Commands\DrushCommands;

/**
 * Drush commands for content completeness index.
 */
final class CompletenessIndexCommands extends DrushCommands {

  /**
   * Constructs a CompletenessIndexCommands object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\content_completeness_index\Service\CompletenessConfigManager $configManager
   *   The config manager.
   * @param \Drupal\content_completeness_index\Service\CompletenessCalculator $calculator
   *   The calculator service.
   * @param \Drupal\content_completeness_index\Service\CompletenessIndexStorage $storage
   *   The storage service.
   */
  public function __construct(
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly CompletenessConfigManager $configManager,
    private readonly CompletenessCalculator $calculator,
    private readonly CompletenessIndexStorage $storage
  ) {
    parent::__construct();
  }

  /**
   * Rebuild completeness indices for all or a specific bundle.
   *
   * @param string|null $bundle
   *   Optional bundle name to limit rebuild to.
   * @param array $options
   *   Command options.
   */
  #[Command(name: 'content-completeness:rebuild', aliases: ['ccr'])]
  #[Argument(name: 'bundle', description: 'The bundle (content type) to rebuild scores for')]
  #[Option(name: 'clear', description: 'Clear existing scores before rebuilding')]
  public function rebuild(?string $bundle = NULL, array $options = ['clear' => FALSE]): void {
    $clear = $options['clear'];
    $bundles = [];

    if ($bundle) {
      // Validate bundle exists and is enabled.
      if (!$this->configManager->isEnabled($bundle)) {
        $this->logger()->error(dt('Bundle "@bundle" does not exist or does not have completeness scoring enabled.', ['@bundle' => $bundle]));
        return;
      }
      $bundles = [$bundle];
    }
    else {
      // Get all enabled bundles.
      $bundles = $this->configManager->getEnabledBundles();
    }

    if (empty($bundles)) {
      $this->logger()->warning(dt('No bundles with completeness scoring enabled found.'));
      return;
    }

    $this->logger()->notice(dt('Rebuilding completeness indices for bundles: @bundles', [
      '@bundles' => implode(', ', $bundles),
    ]));

    $total_processed = 0;

    foreach ($bundles as $bundle_name) {
      $config = $this->configManager->getConfig($bundle_name);
      $weights = $config['weights'];

      // Get all nodes of this bundle.
      $query = $this->entityTypeManager->getStorage('node')->getQuery()
        ->condition('type', $bundle_name)
        ->accessCheck(FALSE);

      $nids = $query->execute();

      if (empty($nids)) {
        $this->logger()->notice(dt('No nodes found for bundle: @bundle', ['@bundle' => $bundle_name]));
        continue;
      }

      $this->logger()->notice(dt('Processing @count nodes for bundle: @bundle', [
        '@count' => count($nids),
        '@bundle' => $bundle_name,
      ]));

      // Process in chunks.
      $chunks = array_chunk($nids, 50);

      foreach ($chunks as $chunk) {
        $nodes = $this->entityTypeManager->getStorage('node')->loadMultiple($chunk);

        foreach ($nodes as $node) {
          $revision_id = $node->getRevisionId();

          // Clear if requested.
          if ($clear) {
            $this->storage->deleteScoresForEntity('node', $node->id());
          }

          // Calculate and save the score.
          $score = $this->calculator->calculate($node, $weights);
          $this->storage->save('node', $node->id(), $revision_id, $score);

          $total_processed++;
        }
      }
    }

    $this->logger()->success(dt('Successfully processed @count entities.', ['@count' => $total_processed]));
  }

  /**
   * Display completeness index statistics.
   */
  #[Command(name: 'content-completeness:stats', aliases: ['ccs'])]
  public function stats(): void {
    $enabled_bundles = $this->configManager->getEnabledBundles();

    if (empty($enabled_bundles)) {
      $this->logger()->warning(dt('No bundles with completeness scoring enabled found.'));
      return;
    }

    $this->logger()->notice(dt('Completeness Index Statistics'));
    $this->logger()->notice(str_repeat('=', 50));

    foreach ($enabled_bundles as $bundle) {
      $config = $this->configManager->getConfig($bundle);

      $query = $this->entityTypeManager->getStorage('node')->getQuery()
        ->condition('type', $bundle)
        ->accessCheck(FALSE);

      $count = $query->count()->execute();

      $this->logger()->notice(dt('Bundle: @bundle', ['@bundle' => $bundle]));
      $this->logger()->notice(dt('  Total nodes: @count', ['@count' => $count]));
      $this->logger()->notice(dt('  Total field weight: @weight', ['@weight' => $config['total_weight']]));
      $this->logger()->notice(dt('  Configured fields: @fields', ['@fields' => count($config['weights'])]));
      $this->logger()->notice('');
    }
  }

}
