<?php

namespace Drupal\content_completeness_index\Hook;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\content_completeness_index\Service\CompletenessCalculator;
use Drupal\content_completeness_index\Service\CompletenessConfigManager;
use Drupal\content_completeness_index\Service\CompletenessIndexStorage;
use Drupal\node\NodeInterface;
use Drupal\node\NodeTypeInterface;

/**
 * Hook implementations for entity operations.
 */
class EntityHooks {

  /**
   * Constructs an EntityHooks object.
   *
   * @param \Drupal\content_completeness_index\Service\CompletenessConfigManager $configManager
   *   The config manager.
   * @param \Drupal\content_completeness_index\Service\CompletenessCalculator $calculator
   *   The calculator service.
   * @param \Drupal\content_completeness_index\Service\CompletenessIndexStorage $storage
   *   The storage service.
   */
  public function __construct(
    private readonly CompletenessConfigManager $configManager,
    private readonly CompletenessCalculator $calculator,
    private readonly CompletenessIndexStorage $storage
  ) {}

  /**
   * Implements hook_ENTITY_TYPE_insert() for node entities.
   */
  #[Hook('node_insert')]
  public function nodeInsert(EntityInterface $entity): void {
    $this->recalculateScoreOnSave($entity);
  }

  /**
   * Implements hook_ENTITY_TYPE_update() for node entities.
   */
  #[Hook('node_update')]
  public function nodeUpdate(EntityInterface $entity): void {
    $this->recalculateScoreOnSave($entity);
  }

  /**
   * Recalculates and saves the completeness index on entity save.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   */
  protected function recalculateScoreOnSave(EntityInterface $entity): void {
    if (!$entity instanceof NodeInterface) {
      return;
    }

    $bundle = $entity->bundle();
    $config = $this->configManager->getConfig($bundle);

    if (!$config['enabled']) {
      return;
    }

    $weights = $config['weights'];
    $score = $this->calculator->calculate($entity, $weights);
    $this->storage->save(
      $entity->getEntityTypeId(),
      $entity->id(),
      $entity->getRevisionId(),
      $score
    );
  }

  /**
   * Implements hook_ENTITY_TYPE_delete() for node entities.
   */
  #[Hook('node_delete')]
  public function nodeDelete(EntityInterface $entity): void {
    if (!$entity instanceof NodeInterface) {
      return;
    }

    // Delete all scores for this entity.
    $this->storage->deleteScoresForEntity(
      $entity->getEntityTypeId(),
      $entity->id()
    );
  }

  /**
   * Implements hook_form_FORM_ID_alter() for node_type_form.
   */
  #[Hook('form_node_type_form_alter')]
  public function formNodeTypeFormAlter(array &$form, FormStateInterface $form_state, string $form_id): void {
    /** @var \Drupal\node\NodeTypeInterface $node_type */
    $node_type = $form_state->getFormObject()->getEntity();
    $bundle = $node_type->id();

    // Get current configuration.
    $config = $this->configManager->getConfig($bundle);

    // Add completeness index settings to the node type form.
    $form['completeness_index'] = [
      '#type' => 'details',
      '#title' => t('Content Completeness Index'),
      '#group' => 'additional_settings',
      '#weight' => 100,
    ];

    $form['completeness_index']['enable_completeness_index'] = [
      '#type' => 'checkbox',
      '#title' => t('Enable content completeness index'),
      '#description' => t('Enable automatic calculation and tracking of content completeness indices for this content type. Configure field weights in the <a href=":url">Completeness Index tab</a>.', [
        ':url' => $bundle ? "/admin/structure/types/manage/{$bundle}/completeness-index" : '#',
      ]),
      '#default_value' => $config['enabled'],
    ];

    $form['completeness_index']['wprogress_description'] = [
      '#type' => 'textarea',
      '#title' => t('Progress widget description'),
      '#description' => t('Text displayed inside the content completeness progress widget. Supports plain text or HTML.'),
      '#default_value' => $config['wprogress_description'] ?? '',
      '#rows' => 4,
      '#states' => [
        'visible' => [
          ':input[name="enable_completeness_index"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // Add custom submit handler.
    $form['#entity_builders'][] = [$this, 'nodeTypeFormBuilderCallback'];
  }

  /**
   * Entity builder callback for node_type_form.
   */
  public function nodeTypeFormBuilderCallback(string $entity_type, NodeTypeInterface $node_type, array &$form, FormStateInterface $form_state): void {
    $bundle = $node_type->id();
    $enabled = (bool) $form_state->getValue('enable_completeness_index');
    $description = trim((string) $form_state->getValue('wprogress_description'));

    // Get existing config.
    $config = $this->configManager->getConfig($bundle);

    // Update enabled status.
    $config['enabled'] = $enabled;
    $config['wprogress_description'] = $description;

    // Save the configuration.
    $this->configManager->saveConfig($bundle, $config);
  }

  /**
   * Implements hook_theme().
   */
  #[Hook('theme')]
  public function theme(): array {
    return [
      'content_completeness_index' => [
        'variables' => [
          'score' => 0,
          'entity' => NULL,
        ],
        'template' => 'content-completeness-index',
      ],
    ];
  }

}
