<?php

namespace Drupal\content_completeness_index\Service;

use Drupal\Core\Database\Connection;

/**
 * Service for storing and retrieving completeness indices.
 */
class CompletenessIndexStorage {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected Connection $database;

  /**
   * Constructs a CompletenessIndexStorage object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   */
  public function __construct(Connection $database) {
    $this->database = $database;
  }

  /**
   * Saves a completeness index for an entity revision.
   *
   * @param string $entity_type
   *   The entity type.
   * @param int $entity_id
   *   The entity ID.
   * @param int $revision_id
   *   The revision ID.
   * @param int $score
   *   The completeness index (0-100).
   */
  public function save(string $entity_type, int $entity_id, int $revision_id, int $score): void {
    $this->database->merge('content_completeness_index')
      ->keys([
        'entity_type' => $entity_type,
        'revision_id' => $revision_id,
      ])
      ->fields([
        'entity_id' => $entity_id,
        'score' => $score,
        'calculated' => \Drupal::time()->getRequestTime(),
      ])
      ->execute();
  }

  /**
   * Gets the score for a specific revision.
   *
   * @param string $entity_type
   *   The entity type.
   * @param int $revision_id
   *   The revision ID.
   *
   * @return int
   *   The completeness index, or 0 if not found.
   */
  public function getScoreForRevision(string $entity_type, int $revision_id): int {
    $result = $this->database->select('content_completeness_index', 'mcs')
      ->fields('mcs', ['score'])
      ->condition('entity_type', $entity_type)
      ->condition('revision_id', $revision_id)
      ->execute()
      ->fetchField();

    return $result !== FALSE ? (int) $result : 0;
  }

  /**
   * Gets the latest score for an entity.
   *
   * @param string $entity_type
   *   The entity type.
   * @param int $entity_id
   *   The entity ID.
   *
   * @return int
   *   The completeness index, or 0 if not found.
   */
  public function getLatestScore(string $entity_type, int $entity_id): int {
    $result = $this->database->select('content_completeness_index', 'mcs')
      ->fields('mcs', ['score'])
      ->condition('entity_type', $entity_type)
      ->condition('entity_id', $entity_id)
      ->orderBy('revision_id', 'DESC')
      ->range(0, 1)
      ->execute()
      ->fetchField();

    return $result !== FALSE ? (int) $result : 0;
  }

  /**
   * Deletes all scores for a specific entity.
   *
   * @param string $entity_type
   *   The entity type.
   * @param int $entity_id
   *   The entity ID.
   */
  public function deleteScoresForEntity(string $entity_type, int $entity_id): void {
    $this->database->delete('content_completeness_index')
      ->condition('entity_type', $entity_type)
      ->condition('entity_id', $entity_id)
      ->execute();
  }

  /**
   * Deletes all scores for a specific bundle.
   *
   * @param string $entity_type
   *   The entity type.
   * @param array $entity_ids
   *   Array of entity IDs for the bundle.
   */
  public function deleteScoresForBundle(string $entity_type, array $entity_ids): void {
    if (empty($entity_ids)) {
      return;
    }

    $this->database->delete('content_completeness_index')
      ->condition('entity_type', $entity_type)
      ->condition('entity_id', $entity_ids, 'IN')
      ->execute();
  }

  /**
   * Checks if a score exists for a revision.
   *
   * @param string $entity_type
   *   The entity type.
   * @param int $revision_id
   *   The revision ID.
   *
   * @return bool
   *   TRUE if a score exists, FALSE otherwise.
   */
  public function scoreExists(string $entity_type, int $revision_id): bool {
    $count = $this->database->select('content_completeness_index', 'mcs')
      ->condition('entity_type', $entity_type)
      ->condition('revision_id', $revision_id)
      ->countQuery()
      ->execute()
      ->fetchField();

    return $count > 0;
  }

}
