(function (Drupal, once) {
  'use strict';

  Drupal.behaviors.contentCompletenessBundleConfig = {
    attach: function (context) {
      const forms = once(
        'content-completeness-bundle-config-form',
        'form#content_completeness_index_bundle_config_form',
        context,
      );

      forms.forEach(function (form) {
        const fields = Array.from(
          form.querySelectorAll('.content-completeness-field'),
        );
        if (!fields.length) {
          return;
        }

        const summaryTotal = form.querySelector(
          '[data-content-completeness-total]',
        );
        const summaryEnabled = form.querySelector(
          '[data-content-completeness-enabled]',
        );

        const updateSummary = function () {
          if (!summaryTotal && !summaryEnabled) {
            return;
          }

          let totalWeight = 0;
          let enabledCount = 0;

          fields.forEach(function (field) {
            const include = field.querySelector(
              'input.content-completeness-include',
            );
            const weightInput = field.querySelector(
              'input.content-completeness-weight',
            );
            if (!include || !weightInput) {
              return;
            }

            if (include.checked) {
              enabledCount += 1;
              const weight = parseFloat(weightInput.value);
              if (!Number.isNaN(weight) && weight > 0) {
                totalWeight += weight;
              }
            }
          });

          if (summaryTotal) {
            summaryTotal.textContent = Number.isInteger(totalWeight)
              ? totalWeight.toString()
              : totalWeight.toFixed(2);
          }
          if (summaryEnabled) {
            summaryEnabled.textContent = enabledCount.toString();
          }
        };

        fields.forEach(function (field) {
          const include = field.querySelector(
            'input.content-completeness-include',
          );
          const weightInput = field.querySelector(
            'input.content-completeness-weight',
          );
          const conditionalLink = field.querySelector(
            '[data-content-completeness-action="conditional-weights"]',
          );

          if (include) {
            const updateState = function () {
              field.classList.toggle('is-disabled', !include.checked);
              if (conditionalLink) {
                conditionalLink.classList.toggle(
                  'is-disabled',
                  !include.checked,
                );
                conditionalLink.setAttribute(
                  'aria-disabled',
                  include.checked ? 'false' : 'true',
                );
              }
              updateSummary();
            };

            updateState();
            include.addEventListener('change', updateState);
          }

          if (conditionalLink) {
            conditionalLink.addEventListener('click', function (event) {
              if (!include || include.checked) {
                return;
              }
              event.preventDefault();
              event.stopPropagation();
            });
          }

          if (weightInput) {
            weightInput.addEventListener('input', updateSummary);
            weightInput.addEventListener('change', updateSummary);
          }
        });

        const toggleFields = function (shouldEnable) {
          fields.forEach(function (field) {
            const include = field.querySelector(
              'input.content-completeness-include',
            );
            if (!include) {
              return;
            }
            if (include.checked === shouldEnable) {
              return;
            }
            include.checked = shouldEnable;
            include.dispatchEvent(
              new Event('change', { bubbles: true, cancelable: true }),
            );
          });
          updateSummary();
        };

        const enableButton = form.querySelector(
          '[data-content-completeness-action="enable-all"]',
        );
        const disableButton = form.querySelector(
          '[data-content-completeness-action="disable-all"]',
        );

        if (enableButton) {
          enableButton.addEventListener('click', function (event) {
            event.preventDefault();
            toggleFields(true);
          });
        }

        if (disableButton) {
          disableButton.addEventListener('click', function (event) {
            event.preventDefault();
            toggleFields(false);
          });
        }

        updateSummary();
      });
    },
  };
})(Drupal, once);
