/**
 * @file
 * JavaScript for Content Completeness Index Floating Form Assistant.
 */

(function (Drupal, drupalSettings, once) {
  'use strict';

  /**
   * Floating form assistant behavior.
   */
  Drupal.behaviors.contentCompletenessFloatingAssistant = {
    attach: function (context, settings) {
      const assistants = once('floating-assistant', '.content-completeness-floating-assistant', context);

      assistants.forEach(function (assistant) {
        const bundle = assistant.dataset.bundle;
        const entityType = assistant.dataset.entityType;
        const entityId = assistant.dataset.entityId || '';
        let weightedFields = [];
        if (assistant.dataset.weightedFields) {
          try {
            const parsed = JSON.parse(assistant.dataset.weightedFields);
            if (Array.isArray(parsed)) {
              weightedFields = parsed;
            }
          }
          catch (e) {
            // Ignore JSON parse errors and fall back to defaults.
          }
        }
        const weightedFieldSet = new Set(weightedFields);
        const scoreDisplays = assistant.querySelectorAll('[data-role="score-display"]');
        const messageElement = assistant.querySelector('.floating-assistant__message');
        const closeButton = assistant.querySelector('.floating-assistant__close');
        const toggleButton = assistant.querySelector('.floating-assistant__toggle');
        const srLabel = toggleButton ? toggleButton.querySelector('.floating-assistant__sr-label') : null;
        const panel = assistant.querySelector('.floating-assistant__panel');
        let currentScore = Number(assistant.dataset.initialScore || 0);
        const startCollapsed = assistant.dataset.startCollapsed === 'true';
        const form = document.querySelector('form.node-form');
        const formId = form ? form.getAttribute('id') : null;
        const cciSettings = drupalSettings.contentCompletenessIndex || {};
        const formSettings = formId && cciSettings.forms ? cciSettings.forms[formId] || null : null;
        const isMultistep = !!(formSettings && formSettings.isMultistep);
        const initialFieldStates = (isMultistep && formSettings && formSettings.initialFieldStates)
          ? formSettings.initialFieldStates
          : {};

        const clampScore = function (score) {
          const numeric = Number(score);
          if (!Number.isFinite(numeric)) {
            return 0;
          }
          return Math.min(100, Math.max(0, Math.round(numeric)));
        };

        const updateToggleLabels = function () {
          if (!toggleButton) {
            return;
          }

          const collapsed = assistant.classList.contains('is-collapsed');
          const tooltip = currentScore + '% complete, open to see more information';
          toggleButton.setAttribute('title', tooltip);
          toggleButton.setAttribute('aria-label', tooltip);
          toggleButton.setAttribute('aria-expanded', collapsed ? 'false' : 'true');
          if (srLabel) {
            srLabel.textContent = tooltip;
          }
        };

        const setCompletion = function (score) {
          currentScore = clampScore(score);
          scoreDisplays.forEach(function (display) {
            display.textContent = currentScore;
          });
          assistant.style.setProperty('--completion', currentScore);
          updateToggleLabels();
        };

        const setCollapsed = function (collapsed) {
          if (collapsed) {
            assistant.classList.add('is-collapsed');
          }
          else {
            assistant.classList.remove('is-collapsed');
          }
          updateToggleLabels();
        };

        // Apply initial state.
        setCompletion(currentScore);
        setCollapsed(startCollapsed);

        if (closeButton) {
          closeButton.addEventListener('click', function () {
            setCollapsed(true);
            if (toggleButton) {
              toggleButton.focus();
            }
          });
        }

        if (toggleButton) {
          toggleButton.addEventListener('click', function () {
            setCollapsed(false);
            if (panel && typeof panel.focus === 'function') {
              panel.focus();
            }
          });
        }

        const createFieldLink = function (field) {
          const link = document.createElement('a');
          link.classList.add('floating-assistant__field-link');

          const strong = document.createElement('strong');
          strong.textContent = field.label || '';
          link.appendChild(strong);

          const anchor = field.anchor ? String(field.anchor) : '';
          if (anchor) {
            link.href = `#${anchor}`;
            link.dataset.fieldAnchor = anchor;
            link.addEventListener('click', function (event) {
              event.preventDefault();

              setCollapsed(true);

              const target = document.getElementById(anchor);
              if (target) {
                target.scrollIntoView({ behavior: 'smooth', block: 'center' });
                const focusable = target.querySelector('input, textarea, select, button');
                if (focusable && typeof focusable.focus === 'function') {
                  focusable.focus({ preventScroll: true });
                }
              }
            });
          }
          else {
            link.href = '#';
            link.addEventListener('click', function (event) {
              event.preventDefault();
            });
          }

          return link;
        };

        const buildMissingMessage = function (groups, ungrouped) {
          if ((!groups || groups.length === 0) && (!ungrouped || ungrouped.length === 0)) {
            const paragraph = document.createElement('p');
            paragraph.textContent = Drupal.t('All weighted fields are complete!');
            return paragraph;
          }

          const wrapper = document.createElement('div');
          wrapper.classList.add('floating-assistant__message-wrapper');

          const intro = document.createElement('p');
          intro.textContent = Drupal.t('Filling the following fields would increase the completeness index:');
          wrapper.appendChild(intro);

          if (groups && groups.length) {
            const groupsContainer = document.createElement('div');
            groupsContainer.classList.add('floating-assistant__groups');

            groups.forEach(function (group) {
              const details = document.createElement('details');
              details.classList.add('floating-assistant__group');
              details.open = false;

              const summary = document.createElement('summary');
              summary.textContent = group.label || '';
              details.appendChild(summary);

              const list = document.createElement('ul');
              list.classList.add('floating-assistant__group-list');

              (group.fields || []).forEach(function (field) {
                const listItem = document.createElement('li');
                listItem.appendChild(createFieldLink(field));
                list.appendChild(listItem);
              });

              details.appendChild(list);
              groupsContainer.appendChild(details);
            });

            wrapper.appendChild(groupsContainer);
          }

          if (ungrouped && ungrouped.length) {
            const ungroupedContainer = document.createElement('div');
            ungroupedContainer.classList.add('floating-assistant__ungrouped');

            const heading = document.createElement('p');
            heading.classList.add('floating-assistant__ungrouped-title');
            heading.textContent = (groups && groups.length)
              ? Drupal.t('Additionally, fill these fields:')
              : Drupal.t('Fill these fields:');
            ungroupedContainer.appendChild(heading);

            const list = document.createElement('ul');
            list.classList.add('floating-assistant__group-list');

            ungrouped.forEach(function (field) {
              const listItem = document.createElement('li');
              listItem.appendChild(createFieldLink(field));
              list.appendChild(listItem);
            });

            ungroupedContainer.appendChild(list);
            wrapper.appendChild(ungroupedContainer);
          }

          return wrapper;
        };
        const renderMissingMessage = function (missingData) {
          if (!messageElement) {
            return;
          }

          let groups = [];
          let ungrouped = [];

          if (missingData && typeof missingData === 'object' && !Array.isArray(missingData)) {
            if (Array.isArray(missingData.groups)) {
              groups = missingData.groups;
            }
            if (Array.isArray(missingData.ungrouped)) {
              ungrouped = missingData.ungrouped;
            }
          }
          else if (Array.isArray(missingData)) {
            ungrouped = missingData;
          }

          messageElement.innerHTML = '';
          const fragment = buildMissingMessage(groups, ungrouped);
          if (fragment) {
            messageElement.appendChild(fragment);
          }
        };

        // Debounce function to limit API calls.
        let debounceTimer;
        const debounce = function (callback, delay) {
          clearTimeout(debounceTimer);
          debounceTimer = setTimeout(callback, delay);
        };

        const cloneFieldStates = function (states) {
          const clone = {};
          Object.keys(states || {}).forEach(function (name) {
            const state = states[name] || {};
            clone[name] = {
              filled: !!state.filled,
              values: Array.isArray(state.values) ? state.values.slice() : [],
            };
          });
          return clone;
        };

        // Function to update the score.
        const updateScore = function () {
          if (!form) {
            return;
          }

          const fieldPayload = isMultistep ? cloneFieldStates(initialFieldStates) : {};
          const trackedFields = new Set(weightedFieldSet);
          if (isMultistep) {
            Object.keys(initialFieldStates).forEach(function (name) {
              trackedFields.add(name);
            });
          }
          const inputs = form.querySelectorAll('input[name], textarea[name], select[name]');
          inputs.forEach(function (input) {
            const match = input.name.match(/^([^\[]+)/);
            if (!match || !match[1]) {
              return;
            }
            const fieldName = match[1];
            if (weightedFieldSet.size === 0) {
              if (!fieldName.startsWith('field_')) {
                return;
              }
            }
            else if (!weightedFieldSet.has(fieldName) && !fieldName.startsWith('field_')) {
              return;
            }

            trackedFields.add(fieldName);
          });

          trackedFields.forEach(function (name) {
            if (!fieldPayload[name]) {
              fieldPayload[name] = { filled: false, values: [] };
            }
          });

          const formData = new FormData(form);
          for (const [inputName, rawValue] of formData.entries()) {
            const match = inputName.match(/^([^\[]+)/);
            if (!match || !match[1]) {
              continue;
            }
            const fieldName = match[1];
            if (!fieldPayload[fieldName]) {
              fieldPayload[fieldName] = { filled: false, values: [] };
            }

            const value = typeof rawValue === 'string' ? rawValue : '';
            if (value !== '') {
              fieldPayload[fieldName].values.push(value);
              fieldPayload[fieldName].filled = true;
            }
          }

          inputs.forEach(function (input) {
            const match = input.name.match(/^([^\[]+)/);
            if (!match || !match[1]) {
              return;
            }
            const fieldName = match[1];
            if (!fieldPayload[fieldName]) {
              fieldPayload[fieldName] = { filled: false, values: [] };
            }

            if ((input.type === 'checkbox' || input.type === 'radio') && input.checked) {
              fieldPayload[fieldName].filled = true;
              if (input.value !== '') {
                fieldPayload[fieldName].values.push(input.value);
              }
            }
            else if (input.type === 'file' && input.files && input.files.length) {
              fieldPayload[fieldName].filled = true;
            }
            else if (!fieldPayload[fieldName].filled) {
              const value = input.value;
              if (typeof value === 'string' && value.trim() !== '') {
                fieldPayload[fieldName].filled = true;
                fieldPayload[fieldName].values.push(value);
              }
            }
          });

          const payload = {
            fields: fieldPayload,
          };

          Object.keys(fieldPayload).forEach(function (fieldName) {
            const fieldState = fieldPayload[fieldName];
            if (Array.isArray(fieldState.values)) {
              fieldState.values = Array.from(new Set(fieldState.values));
            }
          });

          if (entityId) {
            payload.entity_id = entityId;
          }

          // Call preview endpoint.
          const url = Drupal.url(`content-completeness/preview/${entityType}/${bundle}`);
          fetch(url, {
            method: 'POST',
            headers: {
              'Content-Type': 'application/json',
            },
            body: JSON.stringify(payload),
          })
            .then(response => {
              if (!response.ok) {
                return response.text().then(text => {
                  throw new Error(`Request failed with status ${response.status}: ${text}`);
                });
              }
              return response.json();
            })
            .then(data => {
              // Update score display.
              if (data.score !== undefined) {
                setCompletion(data.score);
              }

              // Update missing fields message.
              renderMissingMessage(data.missing);
            })
            .catch(error => {
              console.error('Error updating completeness index:', error);
            });
        };

        // Attach change listeners to form fields.
        if (form) {
          form.addEventListener('change', function () {
            debounce(updateScore, 500);
          });

          form.addEventListener('input', function () {
            debounce(updateScore, 1000);
          });

          // Ensure the score reflects values captured on previous steps.
          updateScore();
        }
      });
    }
  };

})(Drupal, drupalSettings, once);
