<?php

namespace Drupal\content_completeness_index_conditional_validation\Controller;

use Drupal\content_completeness_index_conditional_validation\Service\ConditionalWeightRuleSetManager;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\OpenModalDialogCommand;
use Drupal\Core\Controller\ControllerBase;
use Drupal\node\NodeTypeInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Provides routing callbacks for conditional weight management.
 */
class ConditionalWeightController extends ControllerBase {

  /**
   * Rule set manager service.
   *
   * @var \Drupal\content_completeness_index_conditional_validation\Service\ConditionalWeightRuleSetManager
   */
  protected ConditionalWeightRuleSetManager $ruleSetManager;

  /**
   * Request stack service.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected RequestStack $requestStack;

  /**
   * Constructs the controller.
   */
  public function __construct(ConditionalWeightRuleSetManager $rule_set_manager, RequestStack $request_stack) {
    $this->ruleSetManager = $rule_set_manager;
    $this->requestStack = $request_stack;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('content_completeness_index_conditional_validation.rule_set_manager'),
      $container->get('request_stack')
    );
  }

  /**
   * Opens the Field Validation UI for the conditional rule set in a dialog.
   */
  public function manage(NodeTypeInterface $node_type, string $field_name) {
    try {
      $rule_set = $this->ruleSetManager->ensureRuleSet($node_type, $field_name);
    }
    catch (\InvalidArgumentException $exception) {
      throw new NotFoundHttpException($exception->getMessage(), $exception);
    }

    $request = $this->requestStack->getCurrentRequest();
    if (!$request || !$request->isXmlHttpRequest()) {
      $url = $rule_set->toUrl('edit-form');
      return $this->redirect($url->getRouteName(), $url->getRouteParameters(), $url->getOptions());
    }

    $target_label = $this->ruleSetManager->getTargetFieldLabel($rule_set) ?? $field_name;
    $title = $this->t('Conditional weights: @field', ['@field' => $target_label]);

    $form = $this->entityFormBuilder()->getForm($rule_set, 'edit');

    $response = new AjaxResponse();
    $response->addCommand(new OpenModalDialogCommand($title, $form, ['width' => '75%']));
    return $response;
  }

}
