<?php

namespace Drupal\content_completeness_index_conditional_validation\Plugin\FieldValidationRule;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\field_validation\ConfigurableFieldValidationRuleBase;
use Drupal\field_validation\FieldValidationRuleSetInterface;

/**
 * Provides conditional weighting for the Content Completeness Index.
 *
 * @FieldValidationRule(
 *   id = "cci_conditional_weight_rule",
 *   label = @Translation("CCI conditional weight"),
 *   description = @Translation("Applies a Content Completeness Index weight when the configured condition is met.")
 * )
 */
class ConditionalWeightRule extends ConfigurableFieldValidationRuleBase implements ConditionalWeightRuleInterface {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return parent::defaultConfiguration() + [
      'target_weight' => 0,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form['target_weight'] = [
      '#type' => 'number',
      '#title' => $this->t('Weight to apply'),
      '#description' => $this->t('Weight value used when this rule matches.'),
      '#default_value' => $this->getTargetWeight(),
      '#step' => 0.1,
      '#min' => 0,
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);
    $this->configuration['target_weight'] = (float) $form_state->getValue('target_weight', 0);
  }

  /**
   * {@inheritdoc}
   */
  public function applies(ContentEntityInterface $entity): bool {
    return $this->checkCondition($entity);
  }

  /**
   * {@inheritdoc}
   */
  public function getTargetWeight(): float {
    return isset($this->configuration['target_weight'])
      ? (float) $this->configuration['target_weight']
      : 0.0;
  }

  /**
   * {@inheritdoc}
   */
  public function validate($params) {
    // This rule does not produce validation errors.
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function addFieldValidationRule(FieldValidationRuleSetInterface $field_validation_rule_set) {
    return TRUE;
  }

}
