<?php

namespace Drupal\content_completeness_index_conditional_validation\Service;

use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\field_validation\FieldValidationRuleSetInterface;
use Drupal\field_validation\Entity\FieldValidationRuleSet;
use Drupal\node\NodeTypeInterface;

/**
 * Manages the Field Validation rule sets used for conditional weights.
 */
class ConditionalWeightRuleSetManager {

  use StringTranslationTrait;

  /**
   * The rule set storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $ruleSetStorage;

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected EntityFieldManagerInterface $entityFieldManager;

  /**
   * Constructs the rule set manager.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, EntityFieldManagerInterface $entity_field_manager, TranslationInterface $string_translation) {
    $this->ruleSetStorage = $entity_type_manager->getStorage('field_validation_rule_set');
    $this->entityFieldManager = $entity_field_manager;
    $this->stringTranslation = $string_translation;
  }

  /**
   * Builds the identifier for a conditional rule set.
   */
  public function buildRuleSetId(string $entity_type, string $bundle, string $field_name): string {
    return sprintf('cci__%s__%s__%s', $entity_type, $bundle, $field_name);
  }

  /**
   * Gets the rule set for a given field.
   */
  public function loadRuleSet(string $entity_type, string $bundle, string $field_name): ?FieldValidationRuleSetInterface {
    $id = $this->buildRuleSetId($entity_type, $bundle, $field_name);
    /** @var \Drupal\field_validation\FieldValidationRuleSetInterface|null $rule_set */
    $rule_set = $this->ruleSetStorage->load($id);
    return $rule_set instanceof FieldValidationRuleSetInterface ? $rule_set : NULL;
  }

  /**
   * Ensures a conditional rule set exists for the bundle field.
   */
  public function ensureRuleSet(NodeTypeInterface $node_type, string $field_name): FieldValidationRuleSetInterface {
    $id = $this->buildRuleSetId('node', $node_type->id(), $field_name);
    $field_definitions = $this->entityFieldManager->getFieldDefinitions('node', $node_type->id());
    if (!isset($field_definitions[$field_name])) {
      throw new \InvalidArgumentException(sprintf('Field "%s" does not exist on bundle "%s".', $field_name, $node_type->id()));
    }
    $field_label = $field_definitions[$field_name]->getLabel() ?? $field_name;
    $label = $this->buildRuleSetLabel($node_type, (string) $field_label);

    $rule_set = $this->loadRuleSet('node', $node_type->id(), $field_name);
    if ($rule_set) {
      $rule_set->set('label', $label);
      $rule_set->setThirdPartySetting('content_completeness_index_conditional_validation', 'target_field_label', (string) $field_label);
      $rule_set->save();
      return $rule_set;
    }

    /** @var \Drupal\field_validation\Entity\FieldValidationRuleSet $rule_set */
    $rule_set = FieldValidationRuleSet::create([
      'name' => $id,
      'label' => $label,
      'entity_type' => 'node',
      'bundle' => $node_type->id(),
    ]);
    $rule_set->setThirdPartySetting('content_completeness_index_conditional_validation', 'target_field', $field_name);
    $rule_set->setThirdPartySetting('content_completeness_index_conditional_validation', 'target_field_label', (string) $field_label);
    $rule_set->save();

    return $rule_set;
  }

  /**
   * Builds a label for a new rule set.
   */
  protected function buildRuleSetLabel(NodeTypeInterface $node_type, string $field_label): string {
    return $this->t('CCI conditional weights: @type - @field', [
      '@type' => $node_type->label(),
      '@field' => $field_label,
    ]);
  }

  /**
   * Gets the target field machine name associated with a rule set.
   */
  public function getTargetField(FieldValidationRuleSetInterface $rule_set): ?string {
    return $rule_set->getThirdPartySetting('content_completeness_index_conditional_validation', 'target_field');
  }

  /**
   * Gets the target field label associated with a rule set.
   */
  public function getTargetFieldLabel(FieldValidationRuleSetInterface $rule_set): ?string {
    return $rule_set->getThirdPartySetting('content_completeness_index_conditional_validation', 'target_field_label');
  }

}
