<?php

namespace Drupal\content_completeness_index_field_group\Service;

use Drupal\Component\Utility\Html;
use Drupal\content_completeness_index\Service\FieldGroupingHelperInterface;
use Drupal\Core\Entity\EntityDisplayRepositoryInterface;

/**
 * Helper service for deriving field grouping information.
 */
class FieldGroupingHelper implements FieldGroupingHelperInterface {

  /**
   * Constructs a FieldGroupingHelper object.
   *
   * @param \Drupal\Core\Entity\EntityDisplayRepositoryInterface $entityDisplayRepository
   *   The entity display repository.
   */
  public function __construct(
    private readonly EntityDisplayRepositoryInterface $entityDisplayRepository,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function groupFields(string $entity_type, string $bundle, array $fields): array {
    if (!$fields) {
      return [
        'groups' => [],
        'ungrouped' => [],
      ];
    }

    $group_data = $this->getGroupingData($entity_type, $bundle);
    $field_to_group = $group_data['field_to_group'];
    $top_groups = $group_data['top_groups'];

    $grouped = [];

    // Initialize groups to preserve ordering.
    foreach ($top_groups as $group_info) {
      $grouped[$group_info['label']] = [
        'label' => $group_info['label'],
        'fields' => [],
      ];
    }

    $ungrouped = [];

    foreach ($fields as $field) {
      $field_name = $field['name'] ?? NULL;
      if (!$field_name) {
        continue;
      }

      $group_label = $field_to_group[$field_name] ?? NULL;
      if ($group_label) {
        if (!isset($grouped[$group_label])) {
          $grouped[$group_label] = [
            'label' => $group_label,
            'fields' => [],
          ];
        }
        $grouped[$group_label]['fields'][] = $field;
      }
      else {
        $ungrouped[] = $field;
      }
    }

    $grouped = array_values(array_filter($grouped, static function (array $group): bool {
      return !empty($group['fields']);
    }));

    foreach ($grouped as &$group) {
      $group['fields'] = self::sortFields($group['fields']);
    }
    unset($group);

    $ungrouped = self::sortFields($ungrouped);

    return [
      'groups' => $grouped,
      'ungrouped' => $ungrouped,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormStepMetadata(string $entity_type, string $bundle): array {
    $form_display = $this->entityDisplayRepository->getFormDisplay($entity_type, $bundle, 'default');
    if (!$form_display) {
      return [
        'is_multistep' => FALSE,
        'steps' => [],
      ];
    }

    $group_settings = $form_display->getThirdPartySettings('field_group') ?? [];
    if (!$group_settings) {
      return [
        'is_multistep' => FALSE,
        'steps' => [],
      ];
    }

    $steps = [];

    foreach ($group_settings as $group_name => $definition) {
      if (($definition['format_type'] ?? '') !== 'form_step') {
        continue;
      }

      if (!empty($definition['parent_name'])) {
        continue;
      }

      $label = $this->sanitizeLabel($definition['label'] ?? $group_name);
      $weight = isset($definition['weight']) ? (int) $definition['weight'] : 0;
      $fields = $this->collectGroupFieldNames($group_name, $group_settings);

      $steps[] = [
        'name' => $group_name,
        'label' => $label,
        'weight' => $weight,
        'fields' => $fields,
      ];
    }

    usort($steps, static function (array $a, array $b): int {
      return ($a['weight'] <=> $b['weight'])
        ?: strcmp(mb_strtolower($a['label']), mb_strtolower($b['label']));
    });

    return [
      'is_multistep' => !empty($steps),
      'steps' => $steps,
    ];
  }

  /**
   * Returns the field-to-group mapping and ordered top-level groups.
   */
  protected function getGroupingData(string $entity_type, string $bundle): array {
    $form_display = $this->entityDisplayRepository->getFormDisplay($entity_type, $bundle, 'default');
    if (!$form_display) {
      return [
        'field_to_group' => [],
        'top_groups' => [],
      ];
    }

    $group_settings = $form_display->getThirdPartySettings('field_group') ?? [];
    if (!$group_settings) {
      return [
        'field_to_group' => [],
        'top_groups' => [],
      ];
    }

    $top_groups = [];
    foreach ($group_settings as $group_name => $definition) {
      if (!empty($definition['parent_name'])) {
        continue;
      }

      $label = $this->sanitizeLabel($definition['label'] ?? $group_name);
      $top_groups[$group_name] = [
        'name' => $group_name,
        'label' => $label,
        'weight' => isset($definition['weight']) ? (int) $definition['weight'] : 0,
      ];
    }

    uasort($top_groups, static function (array $a, array $b): int {
      return ($a['weight'] <=> $b['weight'])
        ?: strcmp(mb_strtolower($a['label']), mb_strtolower($b['label']));
    });

    $field_to_group = [];

    foreach ($top_groups as $group_name => $info) {
      $this->mapGroupFields(
        $group_name,
        $group_settings,
        $field_to_group,
        $info['label']
      );
    }

    return [
      'field_to_group' => $field_to_group,
      'top_groups' => array_values($top_groups),
    ];
  }

  /**
   * Recursively assigns fields to top-level group labels.
   */
  protected function mapGroupFields(string $group_name, array $definitions, array &$field_to_group, string $top_label, array &$visited = []): void {
    if (isset($visited[$group_name])) {
      return;
    }

    $visited[$group_name] = TRUE;
    $definition = $definitions[$group_name] ?? [];

    foreach ($definition['children'] ?? [] as $child) {
      if (isset($definitions[$child])) {
        $this->mapGroupFields($child, $definitions, $field_to_group, $top_label, $visited);
      }
      else {
        $field_to_group[$child] = $top_label;
      }
    }

    unset($visited[$group_name]);
  }

  /**
   * Collects all field machine names under a given group.
   */
  protected function collectGroupFieldNames(string $group_name, array $definitions, array &$visited = []): array {
    if (isset($visited[$group_name])) {
      return [];
    }

    $visited[$group_name] = TRUE;
    $definition = $definitions[$group_name] ?? [];
    $fields = [];

    foreach ($definition['children'] ?? [] as $child) {
      if (isset($definitions[$child])) {
        $fields = array_merge($fields, $this->collectGroupFieldNames($child, $definitions, $visited));
      }
      else {
        $fields[] = $child;
      }
    }

    unset($visited[$group_name]);

    return array_values(array_unique($fields));
  }

  /**
   * Normalizes group labels by stripping markup and decoding entities.
   */
  protected function sanitizeLabel(string $label): string {
    $stripped = strip_tags($label);
    return trim(Html::decodeEntities($stripped));
  }

  /**
   * Sorts an array of field metadata.
   */
  protected static function sortFields(array $fields): array {
    usort($fields, static function (array $a, array $b): int {
      $weight_compare = ($a['weight'] ?? 0) <=> ($b['weight'] ?? 0);
      if ($weight_compare !== 0) {
        return $weight_compare;
      }
      return strcmp(
        mb_strtolower((string) ($a['label'] ?? '')),
        mb_strtolower((string) ($b['label'] ?? ''))
      );
    });

    return $fields;
  }

}
