<?php

namespace Drupal\content_completeness_index\Commands;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\content_completeness_index\Service\CompletenessConfigManager;
use Drupal\content_completeness_index\Service\CompletenessRebuildScheduler;
use Drush\Attributes\Command;
use Drush\Commands\DrushCommands;

/**
 * Drush commands for content completeness index.
 */
final class CompletenessIndexCommands extends DrushCommands {

  /**
   * Constructs a CompletenessIndexCommands object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\content_completeness_index\Service\CompletenessConfigManager $configManager
   *   The config manager.
   * @param \Drupal\content_completeness_index\Service\CompletenessRebuildScheduler $rebuildScheduler
   *   Queue scheduler for rebuild jobs.
   */
  public function __construct(
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly CompletenessConfigManager $configManager,
    private readonly CompletenessRebuildScheduler $rebuildScheduler,
  ) {
    parent::__construct();
  }

  /**
   * Rebuild completeness indices for all or a specific bundle.
   *
   * @param string|null $bundle
   *   Optional bundle name to limit rebuild to.
   * @param array $options
   *   Options array.
   *
   * @command content-completeness:rebuild
   *
   * @option clear
   *   Clear existing scores before rebuilding.
   * @option queue-only
   *   Only enqueue work; do not process the queue immediately.
   *
   * @usage drush content-completeness:rebuild
   *   Rebuilds completeness indices for all bundles.
   *
   * @usage drush content-completeness:rebuild article
   *   Rebuilds completeness indices for the "article" bundle only.
   *
   * @usage drush content-completeness:rebuild article --clear
   *   Clears and rebuilds completeness indices for the "article" bundle.
   *
   * @aliases ccr
   */
  public function rebuild(?string $bundle = NULL, array $options = ['clear' => FALSE, 'queue-only' => FALSE]): void {
    $clear = !empty($options['clear']);
    $queue_only = !empty($options['queue-only']);
    $bundles = $this->determineBundles($bundle);

    if (empty($bundles)) {
      $this->logger()->warning(dt('No bundles with completeness scoring enabled found.'));
      return;
    }

    $this->logger()->notice(dt('Queuing completeness rebuild for bundles: @bundles', [
      '@bundles' => implode(', ', $bundles),
    ]));

    $counts = $clear
      ? $this->rebuildScheduler->enqueueBundlesWithReset($bundles)
      : $this->rebuildScheduler->enqueueBundles($bundles);
    foreach ($counts as $bundle_name => $count) {
      $this->logger()->notice(dt('@count @bundle entities enqueued.', [
        '@count' => $count,
        '@bundle' => $bundle_name,
      ]));
    }

    if ($queue_only) {
      $this->logger()->notice(dt('Queue items created. Run "drush queue:run @queue" per bundle or allow cron to process them.', [
        '@queue' => 'content_completeness_index_rebuild:*',
      ]));
      return;
    }

    $processed = $this->rebuildScheduler->processBundles($bundles);
    $total = array_sum($processed);
    $this->logger()->success(dt('Processed @count queued entities.', ['@count' => $total]));
  }

  /**
   * Display completeness index statistics.
   */
  #[Command(name: 'content-completeness:stats', aliases: ['ccs'])]
  public function stats(): void {
    $enabled_bundles = $this->configManager->getEnabledBundles();

    if (empty($enabled_bundles)) {
      $this->logger()->warning(dt('No bundles with completeness scoring enabled found.'));
      return;
    }

    $this->logger()->notice(dt('Completeness Index Statistics'));
    $this->logger()->notice(str_repeat('=', 50));

    foreach ($enabled_bundles as $bundle) {
      $config = $this->configManager->getConfig($bundle);

      $query = $this->entityTypeManager->getStorage('node')->getQuery()
        ->condition('type', $bundle)
        ->accessCheck(FALSE);

      $count = $query->count()->execute();

      $this->logger()->notice(dt('Bundle: @bundle', ['@bundle' => $bundle]));
      $this->logger()->notice(dt('  Total nodes: @count', ['@count' => $count]));
      $this->logger()->notice(dt('  Total field weight: @weight', ['@weight' => $config['total_weight']]));
      $this->logger()->notice(dt('  Configured fields: @fields', ['@fields' => count($config['weights'])]));
      $this->logger()->notice('');
    }
  }

  /**
   * Determines which bundles should be targeted.
   *
   * @return array
   *   Bundle machine names.
   */
  protected function determineBundles(?string $bundle): array {
    if ($bundle) {
      if (!$this->configManager->isEnabled($bundle)) {
        $this->logger()->error(dt('Bundle "@bundle" does not exist or does not have completeness scoring enabled.', ['@bundle' => $bundle]));
        return [];
      }
      return [$bundle];
    }
    return $this->configManager->getEnabledBundles();
  }

}
