<?php

namespace Drupal\content_completeness_index\Form;

use Drupal\content_completeness_index\Form\Helper\CompletenessFieldContextBuilder;
use Drupal\content_completeness_index\Form\Helper\CompletenessSummarySectionBuilder;
use Drupal\content_completeness_index\Form\Helper\CompletenessWeightsSectionBuilder;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\content_completeness_index\Service\CompletenessConfigManager;
use Drupal\content_completeness_index\Service\CompletenessRebuildScheduler;
use Drupal\node\NodeTypeInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configuration form for completeness index per bundle.
 */
class CompletenessBundleConfigForm extends ConfigFormBase {

  /**
   * Builds field context information for the form.
   *
   * @var \Drupal\content_completeness_index\Form\Helper\CompletenessFieldContextBuilder
   */
  protected CompletenessFieldContextBuilder $fieldContextBuilder;

  /**
   * The completeness config manager.
   *
   * @var \Drupal\content_completeness_index\Service\CompletenessConfigManager
   */
  protected CompletenessConfigManager $configManager;

  /**
   * Rebuild scheduler.
   *
   * @var \Drupal\content_completeness_index\Service\CompletenessRebuildScheduler
   */
  protected CompletenessRebuildScheduler $rebuildScheduler;

  /**
   * Builds the weights section of the form.
   *
   * @var \Drupal\content_completeness_index\Form\Helper\CompletenessWeightsSectionBuilder
   */
  protected CompletenessWeightsSectionBuilder $weightsSectionBuilder;

  /**
   * Builds the summary section of the form.
   *
   * @var \Drupal\content_completeness_index\Form\Helper\CompletenessSummarySectionBuilder
   */
  protected CompletenessSummarySectionBuilder $summarySectionBuilder;

  /**
   * The bundle currently being configured.
   *
   * @var string|null
   */
  protected ?string $activeBundle = NULL;

  /**
   * {@inheritdoc}
  */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->configManager = $container->get('content_completeness_index.config_manager');
    $instance->rebuildScheduler = $container->get('content_completeness_index.rebuild_scheduler');
    $instance->fieldContextBuilder = $container->get('content_completeness_index.field_context_builder');
    $instance->weightsSectionBuilder = $container->get('content_completeness_index.weights_section_builder');
    $instance->summarySectionBuilder = $container->get('content_completeness_index.summary_section_builder');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    // This will be dynamically determined based on the bundle.
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'content_completeness_index_bundle_config_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?NodeTypeInterface $node_type = NULL) {
    if (!$node_type) {
      return $form;
    }

    $bundle = $node_type->id();
    $this->activeBundle = $bundle;
    $config = $this->configManager->getConfig($bundle);

    $this->attachFormLibraries($form);
    $form['bundle'] = [
      '#type' => 'value',
      '#value' => $bundle,
    ];
    $form['bulk_actions'] = $this->buildBulkActions();

    $context = $this->fieldContextBuilder->build($bundle);
    $form['weights'] = $this->weightsSectionBuilder->build($context, $config, $this->activeBundle);
    $form['completeness_summary'] = $this->summarySectionBuilder->build($config);

    return parent::buildForm($form, $form_state);
  }

  /**
   * Attaches required libraries for the form UI.
   */
  protected function attachFormLibraries(array &$form): void {
    $form['#attached']['library'][] = 'content_completeness_index/bundle_config_form';
    $form['#attached']['library'][] = 'core/drupal.dialog.ajax';
  }

  /**
   * Builds the bulk action links container.
   */
  protected function buildBulkActions(): array {
    return [
      '#type' => 'container',
      '#weight' => -100,
      '#attributes' => [
        'class' => ['content-completeness-actions'],
      ],
      'enable_all' => [
        '#type' => 'link',
        '#title' => $this->t('Enable all fields'),
        '#url' => Url::fromUserInput('#'),
        '#attributes' => [
          'class' => ['content-completeness-action'],
          'data-content-completeness-action' => 'enable-all',
        ],
      ],
      'disable_all' => [
        '#type' => 'link',
        '#title' => $this->t('Disable all fields'),
        '#url' => Url::fromUserInput('#'),
        '#attributes' => [
          'class' => ['content-completeness-action'],
          'data-content-completeness-action' => 'disable-all',
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $bundle = $form_state->getValue('bundle');
    $config = $this->configManager->getConfig($bundle);
    $enabled = (bool) ($config['enabled'] ?? FALSE);
    $weights = [];
    $field_settings = [];
    $total_weight = 0;

    // Process weights.
    $weight_values = $form_state->getValue('weights');
    if ($weight_values) {
      foreach ($weight_values as $field_name => $field_data) {
        $include = !empty($field_data['include']);
        $weight = isset($field_data['weight']) ? (float) $field_data['weight'] : 0;
        if ($weight < 0) {
          $weight = 0;
        }

        $field_settings[$field_name] = [
          'enabled' => $include,
          'weight' => $weight,
        ];

        if ($include) {
          $weights[$field_name] = $weight;
          if ($weight > 0) {
            $total_weight += $weight;
          }
        }
      }
    }

    $previous_weights = $config['weights'] ?? [];

    // Save configuration.
    $this->configManager->saveConfig($bundle, [
      'enabled' => $enabled,
      'weights' => $weights,
      'field_settings' => $field_settings,
      'total_weight' => $total_weight,
    ]);

    parent::submitForm($form, $form_state);

    $this->messenger()->addStatus($this->t('The completeness index configuration has been saved.'));

    if ($enabled && $weights !== $previous_weights) {
      $queued = $this->rebuildScheduler->enqueueBundleWithReset($bundle);
      $this->messenger()->addStatus($this->t('Queued @count entities for completeness recalculation.', [
        '@count' => $queued,
      ]));
    }
  }

}
