<?php

namespace Drupal\content_completeness_index\Hook;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\content_completeness_index\Service\CompletenessConfigManager;
use Drupal\content_completeness_index\Service\FieldGroupingHelperInterface;
use Drupal\node\NodeInterface;

/**
 * Hook implementations related to form handling.
 */
class FormHooks {

  /**
   * Constructs the form hook subscriber.
   *
   * @param \Drupal\content_completeness_index\Service\CompletenessConfigManager $configManager
   *   The completeness configuration manager.
   * @param \Drupal\content_completeness_index\Service\FieldGroupingHelperInterface $fieldGroupingHelper
   *   Helper for inspecting field group configuration.
   */
  public function __construct(
    private readonly CompletenessConfigManager $configManager,
    private readonly FieldGroupingHelperInterface $fieldGroupingHelper,
  ) {}

  /**
   * Injects multi-step metadata for node edit forms.
   */
  #[Hook('form_node_form_alter')]
  public function formNodeFormAlter(array &$form, FormStateInterface $form_state, string $form_id): void {
    $entity = $this->resolveNodeFromFormState($form_state);
    if (!$entity) {
      return;
    }

    $bundle = $entity->bundle();
    $config = $this->configManager->getConfig($bundle);
    if (!$this->shouldAttachAssistant($config)) {
      return;
    }

    $step_metadata = $this->fieldGroupingHelper->getFormStepMetadata($entity->getEntityTypeId(), $bundle);
    if (!$this->isMultistepWorkflow($step_metadata)) {
      return;
    }

    $form_id_attribute = $this->getFormIdAttribute($form);
    if (!$form_id_attribute) {
      return;
    }

    $weighted_fields = $this->getWeightedFields($config);
    if (!$weighted_fields) {
      return;
    }

    $initial_states = $this->collectInitialStates($entity, $weighted_fields);

    $form['#attached']['drupalSettings']['contentCompletenessIndex']['forms'][$form_id_attribute] = [
      'isMultistep' => TRUE,
      'steps' => $step_metadata['steps'],
      'initialFieldStates' => $initial_states,
      'entityType' => $entity->getEntityTypeId(),
      'bundle' => $bundle,
      'entityId' => $entity->id(),
      'formBuildId' => $form['form_build_id']['#value'] ?? NULL,
    ];
  }

  /**
   * Resolves the node entity from the current form state.
   */
  protected function resolveNodeFromFormState(FormStateInterface $form_state): ?NodeInterface {
    $form_object = $form_state->getFormObject();
    if (!$form_object || !method_exists($form_object, 'getEntity')) {
      return NULL;
    }
    $entity = $form_object->getEntity();
    return $entity instanceof NodeInterface ? $entity : NULL;
  }

  /**
   * Determines whether the assistant should attach based on config.
   */
  protected function shouldAttachAssistant(array $config): bool {
    return !empty($config['enabled']);
  }

  /**
   * Checks whether the form is configured as a multi-step workflow.
   */
  protected function isMultistepWorkflow(array $step_metadata): bool {
    return !empty($step_metadata['is_multistep']) && !empty($step_metadata['steps']);
  }

  /**
   * Gets the form HTML ID if available.
   */
  protected function getFormIdAttribute(array $form): ?string {
    $form_id_attribute = $form['#id'] ?? NULL;
    return is_string($form_id_attribute) && $form_id_attribute !== '' ? $form_id_attribute : NULL;
  }

  /**
   * Returns the list of weighted fields with a positive weight.
   */
  protected function getWeightedFields(array $config): array {
    $weights = $config['weights'] ?? [];
    return array_keys(array_filter($weights, static fn($weight) => (float) $weight > 0));
  }

  /**
   * Collects pre-filled state metadata for weighted fields.
   */
  protected function collectInitialStates(NodeInterface $entity, array $weighted_fields): array {
    $initial_states = [];
    foreach ($weighted_fields as $field_name) {
      if (!$entity->hasField($field_name)) {
        continue;
      }
      $field = $entity->get($field_name);
      if ($field->isEmpty()) {
        continue;
      }
      $initial_states[$field_name] = [
        'filled' => TRUE,
        'values' => $this->extractFieldValues($entity, $field_name),
      ];
    }
    return $initial_states;
  }

  /**
   * Extracts normalized field values from an entity.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity containing the field.
   * @param string $field_name
   *   The field machine name.
   *
   * @return array
   *   A list of scalar values suitable for preview payloads.
   */
  protected function extractFieldValues(ContentEntityInterface $entity, string $field_name): array {
    $field = $entity->get($field_name);
    $values = [];
    $definition = $field->getFieldDefinition();
    $main_property = $this->getMainProperty($definition);

    foreach ($field as $item) {
      if ($main_property !== NULL && isset($item->{$main_property})) {
        $value = $item->{$main_property};
        if ($value !== NULL && $value !== '') {
          $values[] = (string) $value;
        }
        continue;
      }

      foreach ($item->toArray() as $property_value) {
        if (is_scalar($property_value) && $property_value !== '') {
          $values[] = (string) $property_value;
        }
      }
    }

    return array_values(array_unique($values));
  }

  /**
   * Determines the main property name for a field definition.
   */
  protected function getMainProperty(FieldDefinitionInterface $definition): ?string {
    $item_definition = $definition->getItemDefinition();
    $class = $item_definition->getClass();
    if (method_exists($class, 'mainPropertyName')) {
      $property = $class::mainPropertyName();
      if (is_string($property) && $property !== '') {
        return $property;
      }
    }
    return NULL;
  }

}
