<?php

namespace Drupal\content_completeness_index\Plugin\QueueWorker;

use Drupal\content_completeness_index\Service\CompletenessCalculator;
use Drupal\content_completeness_index\Service\CompletenessConfigManager;
use Drupal\content_completeness_index\Service\CompletenessIndexStorage;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\RevisionableInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\QueueWorkerBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Processes completeness rebuild items for a specific bundle.
 *
 * @QueueWorker(
 *   id = "content_completeness_index_rebuild",
 *   title = @Translation("Content completeness index rebuild queue"),
 *   deriver = "Drupal\content_completeness_index\Plugin\Derivative\CompletenessRebuildQueueDeriver"
 * )
 */
class CompletenessRebuildQueueWorker extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  /**
   * Constructs the queue worker.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected readonly CompletenessCalculator $calculator,
    protected readonly CompletenessConfigManager $configManager,
    protected readonly CompletenessIndexStorage $storage,
    protected readonly EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('content_completeness_index.calculator'),
      $container->get('content_completeness_index.config_manager'),
      $container->get('content_completeness_index.storage'),
      $container->get('entity_type.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function processItem($data) {
    $bundle = $data['bundle'] ?? NULL;
    $entity_type = $data['entity_type'] ?? NULL;
    $entity_id = $data['entity_id'] ?? NULL;
    $clear = !empty($data['clear']);

    if (!$bundle || !$entity_type || !$entity_id) {
      return;
    }
    if (!$this->configManager->isEnabled($bundle)) {
      return;
    }

    $storage = $this->entityTypeManager->getStorage($entity_type);
    $entity = $storage->load($entity_id);
    if (!$entity) {
      return;
    }

    if ($clear) {
      $this->storage->deleteScoresForEntity($entity_type, $entity_id);
    }

    $config = $this->configManager->getConfig($bundle);
    $weights = $config['weights'];
    if (!$weights) {
      return;
    }

    $score = $this->calculator->calculate($entity, $weights);
    if ($entity instanceof RevisionableInterface) {
      $this->storage->save($entity_type, $entity_id, $entity->getRevisionId(), $score);
    }
  }

}
