<?php

namespace Drupal\content_completeness_index\Service;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Field\FieldItemListInterface;

/**
 * Service for calculating completeness indices.
 */
class CompletenessCalculator {

  /**
   * Provides conditional weight overrides.
   *
   * @var \Drupal\content_completeness_index\Service\ConditionalWeightEvaluatorInterface
   */
  protected ConditionalWeightEvaluatorInterface $conditionalWeightEvaluator;

  /**
   * Constructs a completeness calculator.
   *
   * @param \Drupal\content_completeness_index\Service\ConditionalWeightEvaluatorInterface $conditional_weight_evaluator
   *   (optional) Conditional weight evaluator service.
   */
  public function __construct(ConditionalWeightEvaluatorInterface $conditional_weight_evaluator) {
    $this->conditionalWeightEvaluator = $conditional_weight_evaluator;
  }

  /**
   * Calculates the completeness index for an entity.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity to calculate the score for.
   * @param array $weights
   *   An associative array of field names to weights.
   *
   * @return int
   *   The completeness index (0-100).
   */
  public function calculate(ContentEntityInterface $entity, array $weights): int {
    if (empty($weights)) {
      return 0;
    }

    $total_weight = 0;
    $filled_weight = 0;

    foreach ($weights as $field_name => $weight) {
      // Skip if weight is 0 or field doesn't exist.
      if ($weight == 0 || !$entity->hasField($field_name)) {
        continue;
      }

      $effective_weight = $this->conditionalWeightEvaluator->getEffectiveWeight($entity, $field_name, (float) $weight);
      if ($effective_weight <= 0) {
        continue;
      }

      $total_weight += $effective_weight;

      // Check if the field is filled.
      if ($this->isFieldFilled($entity->get($field_name))) {
        $filled_weight += $effective_weight;
      }
    }

    // Avoid division by zero.
    if ($total_weight == 0) {
      return 0;
    }

    // Calculate percentage and round to integer.
    return (int) round(($filled_weight / $total_weight) * 100);
  }

  /**
   * Checks if a field is filled (not empty).
   *
   * @param \Drupal\Core\Field\FieldItemListInterface $field
   *   The field to check.
   *
   * @return bool
   *   TRUE if the field has a value, FALSE otherwise.
   */
  protected function isFieldFilled(FieldItemListInterface $field): bool {
    // Check if field is empty.
    if ($field->isEmpty()) {
      return FALSE;
    }

    // For reference fields, check if they have valid references.
    $field_type = $field->getFieldDefinition()->getType();
    if (in_array($field_type, ['entity_reference', 'entity_reference_revisions', 'file', 'image'])) {
      // Check if at least one referenced entity exists.
      foreach ($field as $item) {
        if ($item->entity) {
          return TRUE;
        }
      }
      return FALSE;
    }

    // For text fields, check if there's actual content (not just whitespace).
    if (in_array($field_type, ['string', 'string_long', 'text', 'text_long', 'text_with_summary'])) {
      foreach ($field as $item) {
        $value = $item->value ?? '';
        if (trim($value) !== '') {
          return TRUE;
        }
      }
      return FALSE;
    }

    // For other field types, if not empty, consider it filled.
    return TRUE;
  }

}
