<?php

namespace Drupal\content_completeness_index\Service;

use Drupal\Core\Cache\CacheTagsInvalidatorInterface;
use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * Service for managing completeness index configuration.
 */
class CompletenessConfigManager {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * Cache tag invalidator.
   *
   * @var \Drupal\Core\Cache\CacheTagsInvalidatorInterface
   */
  protected CacheTagsInvalidatorInterface $cacheTagsInvalidator;

  /**
   * Constructs a CompletenessConfigManager object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Cache\CacheTagsInvalidatorInterface $cache_tags_invalidator
   *   Cache tag invalidator service.
   */
  public function __construct(ConfigFactoryInterface $config_factory, CacheTagsInvalidatorInterface $cache_tags_invalidator) {
    $this->configFactory = $config_factory;
    $this->cacheTagsInvalidator = $cache_tags_invalidator;
  }

  /**
   * Gets the configuration for a specific bundle.
   *
   * @param string $bundle
   *   The bundle name.
   *
   * @return array
   *   The configuration array with keys: enabled, weights, total_weight.
   */
  public function getConfig(string $bundle): array {
    $config = $this->configFactory->get("content_completeness_index.settings.{$bundle}");

    $stored_weights = $config->get('weights') ?? [];
    $field_settings = $this->normalizeFieldSettings($stored_weights, $config->get('field_settings') ?? []);
    $weights = $this->extractEnabledWeights($field_settings, $stored_weights);
    $total_weight = $this->calculateTotalWeight($weights);

    return [
      'enabled' => $config->get('enabled') ?? FALSE,
      'weights' => $weights,
      'field_settings' => $field_settings,
      'total_weight' => $total_weight,
      'wprogress_description' => $config->get('wprogress_description') ?? '',
    ];
  }

  /**
   * Saves the configuration for a specific bundle.
   *
   * @param string $bundle
   *   The bundle name.
   * @param array $config
   *   The configuration array.
   */
  public function saveConfig(string $bundle, array $config): void {
    $config_object = $this->configFactory->getEditable("content_completeness_index.settings.{$bundle}");

    $config_object->set('enabled', $config['enabled'] ?? FALSE);
    $config_object->set('weights', $config['weights'] ?? []);
    $config_object->set('field_settings', $config['field_settings'] ?? []);
    $config_object->set('total_weight', $config['total_weight'] ?? 0);
    $config_object->set('wprogress_description', $config['wprogress_description'] ?? '');

    $config_object->save();

    $this->cacheTagsInvalidator->invalidateTags(['content_completeness_index.bundle_list']);
  }

  /**
   * Checks if completeness scoring is enabled for a bundle.
   *
   * @param string $bundle
   *   The bundle name.
   *
   * @return bool
   *   TRUE if enabled, FALSE otherwise.
   */
  public function isEnabled(string $bundle): bool {
    return $this->configFactory->get("content_completeness_index.settings.{$bundle}")->get('enabled') ?? FALSE;
  }

  /**
   * Gets all bundles with completeness scoring enabled.
   *
   * @return array
   *   Array of bundle names.
   */
  public function getEnabledBundles(): array {
    $bundles = [];
    $config_names = $this->configFactory->listAll('content_completeness_index.settings.');

    foreach ($config_names as $config_name) {
      $config = $this->configFactory->get($config_name);
      if ($config->get('enabled')) {
        // Extract bundle name from config name.
        $bundle = str_replace('content_completeness_index.settings.', '', $config_name);
        $bundles[] = $bundle;
      }
    }

    return $bundles;
  }

  /**
   * Normalizes stored field settings, falling back to legacy weights.
   */
  protected function normalizeFieldSettings(array $stored_weights, array $field_settings): array {
    if ($field_settings) {
      return $field_settings;
    }
    $normalized = [];
    foreach ($stored_weights as $field_name => $weight) {
      $normalized[$field_name] = [
        'enabled' => (float) $weight > 0,
        'weight' => $weight,
      ];
    }
    return $normalized;
  }

  /**
   * Extracts the enabled weights list from field settings.
   */
  protected function extractEnabledWeights(array $field_settings, array $stored_weights): array {
    if (!$field_settings) {
      return $stored_weights;
    }

    $weights = [];
    foreach ($field_settings as $field_name => $settings) {
      if (empty($settings['enabled'])) {
        continue;
      }
      $weights[$field_name] = isset($settings['weight']) ? (float) $settings['weight'] : 0;
    }
    return $weights;
  }

  /**
   * Calculates the sum of positive weights.
   */
  protected function calculateTotalWeight(array $weights): float {
    $total = 0.0;
    foreach ($weights as $weight) {
      $value = (float) $weight;
      if ($value > 0) {
        $total += $value;
      }
    }
    return $total;
  }

}
