<?php

namespace Drupal\content_completeness_index\Service;

use Drupal\content_completeness_index\Service\FieldGroupingHelperInterface;
use Drupal\content_completeness_index\Service\Helper\CompletenessPreviewFieldStateProcessor;
use Drupal\content_completeness_index\Service\Helper\CompletenessPreviewWeightCalculator;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;

/**
 * Service for calculating preview scores based on partial field data.
 */
class CompletenessIndexPreviewService {

  /**
   * Constructs a CompletenessIndexPreviewService object.
   *
   * @param \Drupal\content_completeness_index\Service\CompletenessConfigManager $configManager
   *   The config manager.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entityFieldManager
   *   The entity field manager.
   * @param \Drupal\content_completeness_index\Service\FieldGroupingHelperInterface $fieldGroupingHelper
   *   Helper for deriving field grouping information.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\content_completeness_index\Service\Helper\CompletenessPreviewFieldStateProcessor $fieldStateProcessor
   *   Applies and inspects preview field states.
   * @param \Drupal\content_completeness_index\Service\Helper\CompletenessPreviewWeightCalculator $weightCalculator
   *   Calculates weighted preview scores.
   */
  public function __construct(
    private readonly CompletenessConfigManager $configManager,
    private readonly EntityFieldManagerInterface $entityFieldManager,
    private readonly FieldGroupingHelperInterface $fieldGroupingHelper,
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly CompletenessPreviewFieldStateProcessor $fieldStateProcessor,
    private readonly CompletenessPreviewWeightCalculator $weightCalculator,
  ) {}

  /**
   * Calculates a preview score based on field states.
   *
   * @param string $entity_type
   *   The entity type ID.
   * @param string $bundle
   *   The bundle name.
   * @param array $field_states
   *   Array of field names to state data. Each entry can be either a boolean
   *   (legacy format) or an array with the following keys:
   *   - filled: (bool) Whether the field is treated as filled.
   *   - values: (array) Submitted form values used for conditional evaluation.
   * @param int|null $entity_id
   *   (optional) The entity ID to prime data from.
   *
   * @return array
   *   Array with 'score' and 'missing' keys where 'missing' contains:
   *   - groups: Missing fields grouped by their top-level field group.
   *   - ungrouped: Missing fields without a group.
   *
   *   Each missing field entry contains the keys: name, label, weight, anchor.
   */
  public function calculatePreview(string $entity_type, string $bundle, array $field_states, ?int $entity_id = NULL): array {
    $config = $this->configManager->getConfig($bundle);

    if (!$this->isPreviewEnabled($config)) {
      return $this->createEmptyPreview();
    }

    $weights = $config['weights'] ?? [];
    if (!$weights) {
      return $this->createEmptyPreview();
    }

    $field_definitions = $this->entityFieldManager->getFieldDefinitions($entity_type, $bundle);
    $entity = $this->initializeEntity($entity_type, $bundle, $entity_id);
    $this->fieldStateProcessor->applyFieldStates($entity, $field_definitions, $field_states);

    $totals = $this->weightCalculator->summarize($entity, $field_definitions, $weights, $field_states);
    $grouped_missing = $this->fieldGroupingHelper->groupFields($entity_type, $bundle, $totals['missing']);

    return [
      'score' => $totals['score'],
      'missing' => $grouped_missing,
    ];
  }

  /**
   * Determines if preview calculation should run.
   */
  protected function isPreviewEnabled(array $config): bool {
    return !empty($config['enabled']);
  }

  /**
   * Returns an empty preview payload.
   */
  protected function createEmptyPreview(): array {
    return [
      'score' => 0,
      'missing' => [
        'groups' => [],
        'ungrouped' => [],
      ],
    ];
  }

  /**
   * Initializes an entity for preview calculations.
   */
  protected function initializeEntity(string $entity_type, string $bundle, ?int $entity_id): ContentEntityInterface {
    $storage = $this->entityTypeManager->getStorage($entity_type);
    $entity = NULL;

    if ($entity_id) {
      $loaded = $storage->load($entity_id);
      if ($loaded instanceof ContentEntityInterface) {
        $entity = clone $loaded;
      }
    }

    if (!$entity) {
      $entity_definition = $storage->getEntityType();
      $values = [];
      $bundle_key = $entity_definition->getKey('bundle');
      if ($bundle_key) {
        $values[$bundle_key] = $bundle;
      }
      $entity = $storage->create($values);
    }

    return $entity;
  }

}
