<?php

namespace Drupal\content_completeness_index\Service;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Queue\QueueFactory;
use Drupal\Core\Queue\QueueInterface;
use Drupal\Core\Queue\QueueWorkerManagerInterface;
use Psr\Log\LoggerInterface;

/**
 * Coordinates queue-based completeness score rebuilds per bundle.
 */
class CompletenessRebuildScheduler {

  /**
   * Queue worker plugin base ID.
   */
  public const QUEUE_BASE_ID = 'content_completeness_index_rebuild';

  /**
   * The node entity type ID.
   */
  public const ENTITY_TYPE = 'node';

  /**
   * Node storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $nodeStorage;

  /**
   * Logger channel.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * Constructs the scheduler.
   */
  public function __construct(
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    protected readonly QueueFactory $queueFactory,
    protected readonly CompletenessConfigManager $configManager,
    protected readonly QueueWorkerManagerInterface $queueWorkerManager,
    LoggerChannelFactoryInterface $loggerFactory,
  ) {
    $this->nodeStorage = $this->entityTypeManager->getStorage(self::ENTITY_TYPE);
    $this->logger = $loggerFactory->get('content_completeness_index');
  }

  /**
   * Enqueues every entity for the provided bundles.
   *
   * @param array $bundles
   *   Bundle machine names.
   *
   * @return array
   *   Bundle => queued count map.
   */
  public function enqueueBundles(array $bundles): array {
    return $this->scheduleBundles($bundles, FALSE);
  }

  /**
   * Enqueues bundles while clearing stored scores first.
   */
  public function enqueueBundlesWithReset(array $bundles): array {
    return $this->scheduleBundles($bundles, TRUE);
  }

  /**
   * Enqueues all nodes of a bundle for rebuild.
   */
  public function enqueueBundle(string $bundle): int {
    return $this->enqueueBundleInternal($bundle, FALSE);
  }

  /**
   * Enqueues all nodes of a bundle and clears their stored scores first.
   */
  public function enqueueBundleWithReset(string $bundle): int {
    return $this->enqueueBundleInternal($bundle, TRUE);
  }

  /**
   * Shared bundle enqueue helper.
   */
  protected function enqueueBundleInternal(string $bundle, bool $clearExisting): int {
    if (!$this->configManager->isEnabled($bundle)) {
      $this->logger->warning('Skipping completeness queue for bundle "@bundle" because it is not enabled.', [
        '@bundle' => $bundle,
      ]);
      return 0;
    }

    $queue = $this->getQueue($bundle);
    $queue->deleteQueue();

    $nids = $this->nodeStorage->getQuery()
      ->condition('type', $bundle)
      ->accessCheck(FALSE)
      ->execute();

    foreach ($nids as $nid) {
      $queue->createItem([
        'entity_type' => self::ENTITY_TYPE,
        'bundle' => $bundle,
        'entity_id' => (int) $nid,
        'clear' => $clearExisting,
      ]);
    }

    $this->logger->notice('Queued @count entities for @bundle completeness rebuild.', [
      '@count' => count($nids),
      '@bundle' => $bundle,
    ]);

    return count($nids);
  }

  /**
   * Helper for scheduling multiple bundles at once.
   */
  protected function scheduleBundles(array $bundles, bool $clearExisting): array {
    $counts = [];
    foreach ($bundles as $bundle) {
      $counts[$bundle] = $this->enqueueBundleInternal($bundle, $clearExisting);
    }
    return $counts;
  }

  /**
   * Processes queues for the provided bundles until empty.
   *
   * @param array $bundles
   *   Bundle machine names.
   *
   * @return array
   *   Bundle => processed count map.
   */
  public function processBundles(array $bundles): array {
    $counts = [];
    foreach ($bundles as $bundle) {
      $counts[$bundle] = $this->processBundle($bundle);
    }
    return $counts;
  }

  /**
   * Processes a single bundle queue until empty.
   */
  public function processBundle(string $bundle): int {
    $queue_name = $this->getQueueName($bundle);
    if (!$this->queueWorkerManager->hasDefinition($queue_name)) {
      $this->logger->warning('Skipping queue "@queue" because no worker definition exists.', ['@queue' => $queue_name]);
      return 0;
    }

    $queue = $this->getQueue($bundle);
    $worker = $this->queueWorkerManager->createInstance($queue_name);
    $processed = 0;

    while (TRUE) {
      $item = $queue->claimItem();
      if (!$item || !is_object($item)) {
        break;
      }
      try {
        $worker->processItem($item->data);
        $queue->deleteItem($item);
        $processed++;
      }
      catch (\Throwable $throwable) {
        $this->logger->error('Failed processing completeness rebuild queue "@queue": @message', [
          '@queue' => $queue_name,
          '@message' => $throwable->getMessage(),
        ]);
        $queue->releaseItem($item);
        throw $throwable;
      }
    }

    return $processed;
  }

  /**
   * Returns the queue name for a bundle.
   */
  public function getQueueName(string $bundle): string {
    return sprintf('%s:%s', self::QUEUE_BASE_ID, $this->getDerivativeId($bundle));
  }

  /**
   * Gets the derivative ID for the queue worker definition.
   */
  public function getDerivativeId(string $bundle): string {
    return sprintf('%s.%s', self::ENTITY_TYPE, $bundle);
  }

  /**
   * Returns the queue instance for a bundle.
   */
  protected function getQueue(string $bundle): QueueInterface {
    return $this->queueFactory->get($this->getQueueName($bundle));
  }

}
