<?php

namespace Drupal\Tests\content_completeness_index\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;

/**
 * Kernel tests for completeness index calculation and storage.
 *
 * @group content_completeness_index
 */
class CompletenessIndexKernelTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'node',
    'field',
    'text',
    'options',
    'field_validation',
    'content_completeness_index',
    'content_completeness_index_conditional_validation',
  ];

  /**
   * The completeness calculator service.
   *
   * @var \Drupal\content_completeness_index\Service\CompletenessCalculator
   */
  protected $calculator;

  /**
   * The completeness storage service.
   *
   * @var \Drupal\content_completeness_index\Service\CompletenessIndexStorage
   */
  protected $storage;

  /**
   * The completeness config manager service.
   *
   * @var \Drupal\content_completeness_index\Service\CompletenessConfigManager
   */
  protected $configManager;

  /**
   * The rebuild scheduler service.
   *
   * @var \Drupal\content_completeness_index\Service\CompletenessRebuildScheduler
   */
  protected $rebuildScheduler;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('node');
    $this->installSchema('node', ['node_access']);
    $this->installSchema('content_completeness_index', ['content_completeness_index']);
    $this->installConfig(['node', 'field']);

    // Create a content type.
    $node_type = NodeType::create([
      'type' => 'article',
      'name' => 'Article',
    ]);
    $node_type->save();

    FieldStorageConfig::create([
      'field_name' => 'field_domain',
      'entity_type' => 'node',
      'type' => 'list_string',
      'settings' => [
        'allowed_values' => [
          'human' => 'Human',
          'veterinary' => 'Veterinary',
        ],
      ],
      'cardinality' => 1,
    ])->save();
    FieldConfig::create([
      'field_name' => 'field_domain',
      'entity_type' => 'node',
      'bundle' => 'article',
      'label' => 'Domain',
    ])->save();

    FieldStorageConfig::create([
      'field_name' => 'field_species',
      'entity_type' => 'node',
      'type' => 'string',
      'settings' => [
        'max_length' => 255,
      ],
      'cardinality' => 1,
    ])->save();
    FieldConfig::create([
      'field_name' => 'field_species',
      'entity_type' => 'node',
      'bundle' => 'article',
      'label' => 'Species',
    ])->save();

    // Get services.
    $this->calculator = $this->container->get('content_completeness_index.calculator');
    $this->storage = $this->container->get('content_completeness_index.storage');
    $this->configManager = $this->container->get('content_completeness_index.config_manager');
    $this->rebuildScheduler = $this->container->get('content_completeness_index.rebuild_scheduler');
  }

  /**
   * Tests that score is calculated and stored on node save.
   */
  public function testScoreCalculationOnNodeSave(): void {
    // Enable completeness scoring for article content type.
    $this->configManager->saveConfig('article', [
      'enabled' => TRUE,
      'weights' => [
        'title' => 1,
        'body' => 1,
      ],
      'total_weight' => 2,
    ]);

    // Create a node with title only.
    $node = Node::create([
      'type' => 'article',
      'title' => 'Test Article',
    ]);
    $node->save();

    // Check that score was calculated and stored.
    $score = $this->storage->getScoreForRevision('node', $node->getRevisionId());
    $this->assertGreaterThan(0, $score);
    $this->assertLessThanOrEqual(100, $score);

    // The score should be 50% since only title is filled (1 out of 2).
    $this->assertEquals(50, $score);
  }

  /**
   * Tests that score is updated on node update.
   */
  public function testScoreUpdateOnNodeUpdate(): void {
    // Enable completeness scoring.
    $this->configManager->saveConfig('article', [
      'enabled' => TRUE,
      'weights' => [
        'title' => 1,
        'body' => 1,
      ],
      'total_weight' => 2,
    ]);

    // Create a node with title only.
    $node = Node::create([
      'type' => 'article',
      'title' => 'Test Article',
    ]);
    $node->save();

    $initial_score = $this->storage->getScoreForRevision('node', $node->getRevisionId());
    $this->assertEquals(50, $initial_score);

    // Update the node to add body.
    $node->body = [
      'value' => 'Test body content',
      'format' => 'plain_text',
    ];
    $node->save();

    // Check that score was recalculated.
    $updated_score = $this->storage->getScoreForRevision('node', $node->getRevisionId());
    $this->assertEquals(100, $updated_score);
  }

  /**
   * Tests queue-based rebuild scheduling and processing.
   */
  public function testQueueRebuildProcessing(): void {
    $this->configManager->saveConfig('article', [
      'enabled' => TRUE,
      'weights' => [
        'title' => 1,
      ],
      'total_weight' => 1,
    ]);

    $node = Node::create([
      'type' => 'article',
      'title' => 'Initially complete',
    ]);
    $node->save();

    $initial_score = $this->storage->getScoreForRevision('node', $node->getRevisionId());
    $this->assertEquals(100, $initial_score);

    // Make species required by giving it a positive weight.
    $this->configManager->saveConfig('article', [
      'enabled' => TRUE,
      'weights' => [
        'title' => 1,
        'field_species' => 1,
      ],
      'total_weight' => 2,
    ]);

    $this->rebuildScheduler->enqueueBundleWithReset('article');
    $this->rebuildScheduler->processBundle('article');

    $updated_score = $this->storage->getLatestScore('node', $node->id());
    $this->assertEquals(50, $updated_score);
  }

  /**
   * Tests that no score is saved when completeness is disabled.
   */
  public function testNoScoreWhenDisabled(): void {
    // Do not enable completeness scoring.
    $this->configManager->saveConfig('article', [
      'enabled' => FALSE,
      'weights' => [],
      'total_weight' => 0,
    ]);

    // Create a node.
    $node = Node::create([
      'type' => 'article',
      'title' => 'Test Article',
    ]);
    $node->save();

    // Check that no score was saved.
    $score = $this->storage->getScoreForRevision('node', $node->getRevisionId());
    $this->assertEquals(0, $score);
  }

  /**
   * Tests score calculation with different field weights.
   */
  public function testScoreWithDifferentWeights(): void {
    // Enable with different weights.
    $this->configManager->saveConfig('article', [
      'enabled' => TRUE,
      'weights' => [
        'title' => 2,
        'body' => 1,
      ],
      'total_weight' => 3,
    ]);

    // Create a node with title only.
    $node = Node::create([
      'type' => 'article',
      'title' => 'Test Article',
    ]);
    $node->save();

    // Score should be 67% (2 out of 3).
    $score = $this->storage->getScoreForRevision('node', $node->getRevisionId());
    $this->assertEquals(67, $score);
  }

  /**
   * Tests conditional weight rules are applied from Field Validation.
   */
  public function testConditionalWeightIntegration(): void {
    $this->configManager->saveConfig('article', [
      'enabled' => TRUE,
      'weights' => [
        'title' => 1,
        'field_species' => 0,
      ],
      'total_weight' => 1,
    ]);

    /** @var \Drupal\content_completeness_index_conditional_validation\Service\ConditionalWeightRuleSetManager $rule_set_manager */
    $rule_set_manager = $this->container->get('content_completeness_index_conditional_validation.rule_set_manager');
    $node_type = NodeType::load('article');
    $rule_set = $rule_set_manager->ensureRuleSet($node_type, 'field_species');

    // Reset any existing rules before adding test rule.
    $rule_set->set('field_validation_rules', []);
    $rule_set->save();

    $configuration = [
      'uuid' => $this->container->get('uuid')->generate(),
      'id' => 'cci_conditional_weight_rule',
      'title' => 'Species required for veterinary domain',
      'weight' => 0,
      'field_name' => 'field_species',
      'column' => 'value',
      'error_message' => '',
      'roles' => [],
      'condition' => [
        'field' => 'field_domain',
        'operator' => 'equals',
        'value' => 'veterinary',
      ],
      'data' => [
        'target_weight' => 1,
      ],
    ];
    $rule_set->addFieldValidationRule($configuration);
    $rule_set->save();

    // Human domain: conditional weight ignored.
    $node_human = Node::create([
      'type' => 'article',
      'title' => 'Human abstract',
      'field_domain' => ['value' => 'human'],
    ]);
    $node_human->save();
    $score_human = $this->storage->getScoreForRevision('node', $node_human->getRevisionId());
    $this->assertEquals(100, $score_human);

    // Veterinary domain without species: conditional weight applies.
    // Field stays empty so the score drops.
    $node_vet_missing = Node::create([
      'type' => 'article',
      'title' => 'Vet abstract',
      'field_domain' => ['value' => 'veterinary'],
    ]);
    $node_vet_missing->save();
    $score_vet_missing = $this->storage->getScoreForRevision('node', $node_vet_missing->getRevisionId());
    $this->assertEquals(50, $score_vet_missing);

    // Veterinary domain with species completes all weighted fields.
    $node_vet = Node::create([
      'type' => 'article',
      'title' => 'Vet abstract completed',
      'field_domain' => ['value' => 'veterinary'],
      'field_species' => ['value' => 'Canine'],
    ]);
    $node_vet->save();
    $score_vet = $this->storage->getScoreForRevision('node', $node_vet->getRevisionId());
    $this->assertEquals(100, $score_vet);
  }

  /**
   * Tests that entity revision is not duplicated.
   */
  public function testNoRevisionDuplication(): void {
    // Enable completeness scoring.
    $this->configManager->saveConfig('article', [
      'enabled' => TRUE,
      'weights' => [
        'title' => 1,
      ],
      'total_weight' => 1,
    ]);

    // Create a node.
    $node = Node::create([
      'type' => 'article',
      'title' => 'Test Article',
    ]);
    $node->save();

    $initial_revision_id = $node->getRevisionId();

    // Update without creating new revision.
    $node->setNewRevision(FALSE);
    $node->title = 'Updated Title';
    $node->save();

    // Revision ID should remain the same.
    $this->assertEquals($initial_revision_id, $node->getRevisionId());
  }

  /**
   * Tests score retrieval methods.
   */
  public function testScoreRetrieval(): void {
    // Enable completeness scoring.
    $this->configManager->saveConfig('article', [
      'enabled' => TRUE,
      'weights' => [
        'title' => 1,
      ],
      'total_weight' => 1,
    ]);

    // Create a node.
    $node = Node::create([
      'type' => 'article',
      'title' => 'Test Article',
    ]);
    $node->save();

    // Test getScoreForRevision.
    $score_by_revision = $this->storage->getScoreForRevision('node', $node->getRevisionId());
    $this->assertEquals(100, $score_by_revision);

    // Test getLatestScore.
    $latest_score = $this->storage->getLatestScore('node', $node->id());
    $this->assertEquals(100, $latest_score);

    // Test scoreExists.
    $exists = $this->storage->scoreExists('node', $node->getRevisionId());
    $this->assertTrue($exists);
  }

  /**
   * Tests score deletion on node deletion.
   */
  public function testScoreDeletionOnNodeDelete(): void {
    // Enable completeness scoring.
    $this->configManager->saveConfig('article', [
      'enabled' => TRUE,
      'weights' => [
        'title' => 1,
      ],
      'total_weight' => 1,
    ]);

    // Create a node.
    $node = Node::create([
      'type' => 'article',
      'title' => 'Test Article',
    ]);
    $node->save();

    $revision_id = $node->getRevisionId();

    // Verify score exists.
    $this->assertTrue($this->storage->scoreExists('node', $revision_id));

    // Delete the node.
    $node->delete();

    // Verify score was deleted.
    $this->assertFalse($this->storage->scoreExists('node', $revision_id));
  }

}
