<?php

namespace Drupal\content_completeness_score\Form;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\content_completeness_score\Service\CompletenessCalculator;
use Drupal\content_completeness_score\Service\CompletenessConfigManager;
use Drupal\content_completeness_score\Service\CompletenessScoreStorage;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Batch form for recalculating completeness scores.
 */
class CompletenessBatchForm extends FormBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The completeness config manager.
   *
   * @var \Drupal\content_completeness_score\Service\CompletenessConfigManager
   */
  protected CompletenessConfigManager $configManager;

  /**
   * The completeness calculator.
   *
   * @var \Drupal\content_completeness_score\Service\CompletenessCalculator
   */
  protected CompletenessCalculator $calculator;

  /**
   * The completeness score storage.
   *
   * @var \Drupal\content_completeness_score\Service\CompletenessScoreStorage
   */
  protected CompletenessScoreStorage $storage;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = new static();
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->configManager = $container->get('content_completeness_score.config_manager');
    $instance->calculator = $container->get('content_completeness_score.calculator');
    $instance->storage = $container->get('content_completeness_score.storage');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'content_completeness_score_batch_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $enabled_bundles = $this->configManager->getEnabledBundles();

    if (empty($enabled_bundles)) {
      $form['message'] = [
        '#markup' => $this->t('No content types have completeness scoring enabled. Please enable it on a content type first.'),
      ];
      return $form;
    }

    $form['description'] = [
      '#markup' => $this->t('<p>Use this form to recalculate completeness scores for all content with enabled scoring.</p><p><strong>Enabled content types:</strong> @bundles</p>', [
        '@bundles' => implode(', ', $enabled_bundles),
      ]),
    ];

    $form['calculate_missing'] = [
      '#type' => 'submit',
      '#value' => $this->t('Calculate missing scores'),
      '#submit' => ['::submitCalculateMissing'],
    ];

    $form['recalculate_all'] = [
      '#type' => 'submit',
      '#value' => $this->t('Clear and recalculate all scores'),
      '#submit' => ['::submitRecalculateAll'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // This is not used as we have specific submit handlers.
  }

  /**
   * Submit handler for calculating missing scores.
   */
  public function submitCalculateMissing(array &$form, FormStateInterface $form_state) {
    $operations = $this->buildBatchOperations(FALSE);

    if (empty($operations)) {
      $this->messenger()->addWarning($this->t('No missing scores found.'));
      return;
    }

    $batch = [
      'title' => $this->t('Calculating missing completeness scores...'),
      'operations' => $operations,
      'finished' => [static::class, 'batchFinished'],
      'progress_message' => $this->t('Processed @current out of @total entities.'),
    ];

    batch_set($batch);
  }

  /**
   * Submit handler for recalculating all scores.
   */
  public function submitRecalculateAll(array &$form, FormStateInterface $form_state) {
    $operations = $this->buildBatchOperations(TRUE);

    if (empty($operations)) {
      $this->messenger()->addWarning($this->t('No content found to process.'));
      return;
    }

    $batch = [
      'title' => $this->t('Recalculating all completeness scores...'),
      'operations' => $operations,
      'finished' => [static::class, 'batchFinished'],
      'progress_message' => $this->t('Processed @current out of @total entities.'),
    ];

    batch_set($batch);
  }

  /**
   * Builds batch operations for recalculating scores.
   *
   * @param bool $recalculate_all
   *   Whether to recalculate all scores or only missing ones.
   *
   * @return array
   *   Array of batch operations.
   */
  protected function buildBatchOperations(bool $recalculate_all): array {
    $operations = [];
    $enabled_bundles = $this->configManager->getEnabledBundles();

    foreach ($enabled_bundles as $bundle) {
      // Get all nodes of this bundle.
      $query = $this->entityTypeManager->getStorage('node')->getQuery()
        ->condition('type', $bundle)
        ->accessCheck(FALSE);

      $nids = $query->execute();

      if (empty($nids)) {
        continue;
      }

      // Process in chunks of 20.
      $chunks = array_chunk($nids, 20);

      foreach ($chunks as $chunk) {
        $operations[] = [
          [static::class, 'batchProcess'],
          [$chunk, $bundle, $recalculate_all],
        ];
      }
    }

    return $operations;
  }

  /**
   * Batch process callback.
   *
   * @param array $nids
   *   Array of node IDs to process.
   * @param string $bundle
   *   The bundle name.
   * @param bool $recalculate_all
   *   Whether to recalculate all or only missing.
   * @param array $context
   *   Batch context array.
   */
  public static function batchProcess(array $nids, string $bundle, bool $recalculate_all, array &$context) {
    $entity_type_manager = \Drupal::entityTypeManager();
    $config_manager = \Drupal::service('content_completeness_score.config_manager');
    $calculator = \Drupal::service('content_completeness_score.calculator');
    $storage = \Drupal::service('content_completeness_score.storage');

    $config = $config_manager->getConfig($bundle);
    $weights = $config['weights'];

    $nodes = $entity_type_manager->getStorage('node')->loadMultiple($nids);

    foreach ($nodes as $node) {
      $revision_id = $node->getRevisionId();

      // Skip if not recalculating all and score exists.
      if (!$recalculate_all && $storage->scoreExists('node', $revision_id)) {
        continue;
      }

      // Calculate and save the score.
      $score = $calculator->calculate($node, $weights);
      $storage->save('node', $node->id(), $revision_id, $score);

      $context['results'][] = $node->id();
    }

    $context['message'] = t('Processing bundle: @bundle', ['@bundle' => $bundle]);
  }

  /**
   * Batch finished callback.
   *
   * @param bool $success
   *   Whether the batch completed successfully.
   * @param array $results
   *   The results array.
   * @param array $operations
   *   Remaining operations.
   */
  public static function batchFinished(bool $success, array $results, array $operations) {
    if ($success) {
      $count = count($results);
      \Drupal::messenger()->addStatus(t('Successfully processed @count entities.', ['@count' => $count]));
    }
    else {
      \Drupal::messenger()->addError(t('An error occurred during processing.'));
    }
  }

}
