<?php

namespace Drupal\content_completeness_score\Service;

use Drupal\Component\Utility\Html;
use Drupal\Core\Entity\EntityDisplayRepositoryInterface;

/**
 * Helper service for deriving field grouping information.
 */
class FieldGroupingHelper {

  /**
   * Constructs a FieldGroupingHelper object.
   *
   * @param \Drupal\Core\Entity\EntityDisplayRepositoryInterface $entityDisplayRepository
   *   The entity display repository.
   */
  public function __construct(
    private readonly EntityDisplayRepositoryInterface $entityDisplayRepository,
  ) {}

  /**
   * Builds grouped/ungrouped collections for the provided fields.
   *
   * The $fields array must contain associative arrays with at least the keys:
   * - name: The field machine name.
   * - label: A human readable label.
   * - weight: A numeric weight used for ordering.
   *
   * @param string $entity_type
   *   The entity type ID.
   * @param string $bundle
   *   The bundle machine name.
   * @param array $fields
   *   The list of field metadata arrays to group.
   *
   * @return array
   *   An array with:
   *   - groups: Array of groups, each with 'label' and ordered 'fields'.
   *   - ungrouped: Array of fields that do not belong to a group.
   */
  public function groupFields(string $entity_type, string $bundle, array $fields): array {
    if (!$fields) {
      return [
        'groups' => [],
        'ungrouped' => [],
      ];
    }

    $group_data = $this->getGroupingData($entity_type, $bundle);
    $field_to_group = $group_data['field_to_group'];
    $top_groups = $group_data['top_groups'];

    $grouped = [];

    // Initialize groups to preserve ordering.
    foreach ($top_groups as $group_info) {
      $grouped[$group_info['label']] = [
        'label' => $group_info['label'],
        'fields' => [],
      ];
    }

    $ungrouped = [];

    foreach ($fields as $field) {
      $field_name = $field['name'] ?? NULL;
      if (!$field_name) {
        continue;
      }

      $group_label = $field_to_group[$field_name] ?? NULL;
      if ($group_label) {
        if (!isset($grouped[$group_label])) {
          $grouped[$group_label] = [
            'label' => $group_label,
            'fields' => [],
          ];
        }
        $grouped[$group_label]['fields'][] = $field;
      }
      else {
        $ungrouped[] = $field;
      }
    }

    // Filter empty groups and order their fields.
    $grouped = array_values(array_filter($grouped, static function (array $group): bool {
      return !empty($group['fields']);
    }));

    foreach ($grouped as &$group) {
      $group['fields'] = self::sortFields($group['fields']);
    }
    unset($group);

    $ungrouped = self::sortFields($ungrouped);

    return [
      'groups' => $grouped,
      'ungrouped' => $ungrouped,
    ];
  }

  /**
   * Returns the field-to-group mapping and ordered top-level groups.
   *
   * @param string $entity_type
   *   The entity type ID.
   * @param string $bundle
   *   The bundle machine name.
   *
   * @return array
   *   An array with:
   *   - field_to_group: Associative array mapping field name => top group label.
   *   - top_groups: Ordered array of top group metadata (name, label, weight).
   */
  protected function getGroupingData(string $entity_type, string $bundle): array {
    $form_display = $this->entityDisplayRepository->getFormDisplay($entity_type, $bundle, 'default');
    if (!$form_display) {
      return [
        'field_to_group' => [],
        'top_groups' => [],
      ];
    }

    $group_settings = $form_display->getThirdPartySettings('field_group') ?? [];
    if (!$group_settings) {
      return [
        'field_to_group' => [],
        'top_groups' => [],
      ];
    }

    $top_groups = [];
    foreach ($group_settings as $group_name => $definition) {
      if (!empty($definition['parent_name'])) {
        continue;
      }

      $label = $this->sanitizeLabel($definition['label'] ?? $group_name);
      $top_groups[$group_name] = [
        'name' => $group_name,
        'label' => $label,
        'weight' => isset($definition['weight']) ? (int) $definition['weight'] : 0,
      ];
    }

    // Sort top-level groups by weight then label to mirror form display order.
    uasort($top_groups, static function (array $a, array $b): int {
      return ($a['weight'] <=> $b['weight'])
        ?: strcmp(mb_strtolower($a['label']), mb_strtolower($b['label']));
    });

    $field_to_group = [];

    foreach ($top_groups as $group_name => $info) {
      $this->mapGroupFields(
        $group_name,
        $group_settings,
        $field_to_group,
        $info['label']
      );
    }

    return [
      'field_to_group' => $field_to_group,
      'top_groups' => array_values($top_groups),
    ];
  }

  /**
   * Recursively assigns fields to the provided top group label.
   *
   * @param string $group_name
   *   The current group machine name.
   * @param array $definitions
   *   All group definitions keyed by machine name.
   * @param array $field_to_group
   *   Reference to the mapping array being built.
   * @param string $top_label
   *   The label of the top-level group these fields belong to.
   * @param array $visited
   *   Tracks visited groups to avoid infinite recursion.
   */
  protected function mapGroupFields(string $group_name, array $definitions, array &$field_to_group, string $top_label, array &$visited = []): void {
    if (isset($visited[$group_name])) {
      return;
    }

    $visited[$group_name] = TRUE;
    $definition = $definitions[$group_name] ?? [];

    foreach ($definition['children'] ?? [] as $child) {
      if (isset($definitions[$child])) {
        $this->mapGroupFields($child, $definitions, $field_to_group, $top_label, $visited);
      }
      else {
        $field_to_group[$child] = $top_label;
      }
    }

    unset($visited[$group_name]);
  }

  /**
   * Normalizes group labels by stripping markup and decoding entities.
   *
   * @param string $label
   *   Raw label value.
   *
   * @return string
   *   Sanitized label.
   */
  protected function sanitizeLabel(string $label): string {
    $stripped = strip_tags($label);
    return trim(Html::decodeEntities($stripped));
  }

  /**
   * Sorts an array of field metadata.
   *
   * @param array $fields
   *   The fields to sort.
   *
   * @return array
   *   The sorted array.
   */
  protected static function sortFields(array $fields): array {
    usort($fields, static function (array $a, array $b): int {
      $weight_compare = ($a['weight'] ?? 0) <=> ($b['weight'] ?? 0);
      if ($weight_compare !== 0) {
        return $weight_compare;
      }
      return strcmp(
        mb_strtolower((string) ($a['label'] ?? '')),
        mb_strtolower((string) ($b['label'] ?? ''))
      );
    });

    return $fields;
  }

}
