<?php

namespace Drupal\content_completeness_score\Service;

use Drupal\Core\Entity\EntityFieldManagerInterface;

/**
 * Service for calculating preview scores based on partial field data.
 */
class CompletenessScorePreviewService {

  /**
   * Constructs a CompletenessScorePreviewService object.
   *
   * @param \Drupal\content_completeness_score\Service\CompletenessConfigManager $configManager
   *   The config manager.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entityFieldManager
   *   The entity field manager.
   * @param \Drupal\content_completeness_score\Service\FieldGroupingHelper $fieldGroupingHelper
   *   Helper for deriving field grouping information.
   */
  public function __construct(
    private readonly CompletenessConfigManager $configManager,
    private readonly EntityFieldManagerInterface $entityFieldManager,
    private readonly FieldGroupingHelper $fieldGroupingHelper,
  ) {}

  /**
   * Calculates a preview score based on field states.
   *
   * @param string $entity_type
   *   The entity type ID.
   * @param string $bundle
   *   The bundle name.
   * @param array $field_states
   *   Array of field names => boolean (filled or not).
   *
   * @return array
   *   Array with 'score' and 'missing' keys where 'missing' contains:
   *   - groups: Missing fields grouped by their top-level field group.
   *   - ungrouped: Missing fields without a group.
   *
   *   Each missing field entry contains the keys: name, label, weight, anchor.
   */
  public function calculatePreview(string $entity_type, string $bundle, array $field_states): array {
    $config = $this->configManager->getConfig($bundle);

    if (!$config['enabled']) {
      return [
        'score' => 0,
        'missing' => [
          'groups' => [],
          'ungrouped' => [],
        ],
      ];
    }

    $weights = $config['weights'] ?? [];
    $field_definitions = $this->entityFieldManager->getFieldDefinitions($entity_type, $bundle);

    $total_weight = 0;
    $filled_weight = 0;
    $missing = [];

    foreach ($weights as $field_name => $weight) {
      if ($weight <= 0) {
        continue;
      }

      $total_weight += $weight;

      // Check if field is filled in the preview data.
      $is_filled = !empty($field_states[$field_name]);

      if ($is_filled) {
        $filled_weight += $weight;
      }
      else {
        // Add to missing fields.
        $field_definition = $field_definitions[$field_name] ?? NULL;
        if ($field_definition) {
          $missing[] = [
            'name' => $field_name,
            'label' => (string) $field_definition->getLabel(),
            'weight' => $weight,
            'anchor' => 'edit-' . strtr($field_name, ['_' => '-']) . '-wrapper',
          ];
        }
      }
    }

    // Calculate score.
    $score = $total_weight > 0 ? (int) round(($filled_weight / $total_weight) * 100) : 0;

    $grouped_missing = $this->fieldGroupingHelper->groupFields($entity_type, $bundle, $missing);

    return [
      'score' => $score,
      'missing' => $grouped_missing,
    ];
  }

}
