<?php

namespace Drupal\Tests\content_completeness_score\Unit;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\content_completeness_score\Service\CompletenessCalculator;
use Drupal\Tests\UnitTestCase;

/**
 * Unit tests for CompletenessCalculator service.
 *
 * @coversDefaultClass \Drupal\content_completeness_score\Service\CompletenessCalculator
 * @group content_completeness_score
 */
class CompletenessCalculatorTest extends UnitTestCase {

  /**
   * The completeness calculator service.
   *
   * @var \Drupal\content_completeness_score\Service\CompletenessCalculator
   */
  protected CompletenessCalculator $calculator;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->calculator = new CompletenessCalculator();
  }

  /**
   * Tests calculation with empty weights.
   *
   * @covers ::calculate
   */
  public function testCalculateWithEmptyWeights(): void {
    $entity = $this->createMockEntity([]);
    $score = $this->calculator->calculate($entity, []);

    $this->assertEquals(0, $score);
  }

  /**
   * Tests calculation with all fields filled.
   *
   * @covers ::calculate
   */
  public function testCalculateWithAllFieldsFilled(): void {
    $entity = $this->createMockEntity([
      'field_title' => TRUE,
      'field_description' => TRUE,
      'field_author' => TRUE,
    ]);

    $weights = [
      'field_title' => 1,
      'field_description' => 1,
      'field_author' => 1,
    ];

    $score = $this->calculator->calculate($entity, $weights);
    $this->assertEquals(100, $score);
  }

  /**
   * Tests calculation with partially filled fields.
   *
   * @covers ::calculate
   */
  public function testCalculateWithPartiallyFilledFields(): void {
    $entity = $this->createMockEntity([
      'field_title' => TRUE,
      'field_description' => FALSE,
      'field_author' => TRUE,
    ]);

    $weights = [
      'field_title' => 1,
      'field_description' => 1,
      'field_author' => 1,
    ];

    $score = $this->calculator->calculate($entity, $weights);
    // 2 out of 3 fields filled = 67% (rounded)
    $this->assertEquals(67, $score);
  }

  /**
   * Tests calculation with different weights.
   *
   * @covers ::calculate
   */
  public function testCalculateWithDifferentWeights(): void {
    $entity = $this->createMockEntity([
      'field_title' => TRUE,
      'field_description' => FALSE,
      'field_author' => TRUE,
    ]);

    $weights = [
      'field_title' => 2,
      'field_description' => 1,
      'field_author' => 1,
    ];

    $score = $this->calculator->calculate($entity, $weights);
    // (2 + 1) / (2 + 1 + 1) * 100 = 75%
    $this->assertEquals(75, $score);
  }

  /**
   * Tests calculation with zero-weight fields.
   *
   * @covers ::calculate
   */
  public function testCalculateWithZeroWeightFields(): void {
    $entity = $this->createMockEntity([
      'field_title' => TRUE,
      'field_description' => FALSE,
      'field_author' => TRUE,
    ]);

    $weights = [
      'field_title' => 1,
      'field_description' => 0,
      'field_author' => 1,
    ];

    $score = $this->calculator->calculate($entity, $weights);
    // field_description is ignored due to zero weight
    // 2 out of 2 = 100%
    $this->assertEquals(100, $score);
  }

  /**
   * Tests calculation with fractional weights.
   *
   * @covers ::calculate
   */
  public function testCalculateWithFractionalWeights(): void {
    $entity = $this->createMockEntity([
      'field_title' => TRUE,
      'field_description' => TRUE,
      'field_author' => FALSE,
    ]);

    $weights = [
      'field_title' => 0.5,
      'field_description' => 1.5,
      'field_author' => 1,
    ];

    $score = $this->calculator->calculate($entity, $weights);
    // (0.5 + 1.5) / (0.5 + 1.5 + 1) * 100 = 66.67% = 67% rounded
    $this->assertEquals(67, $score);
  }

  /**
   * Tests calculation with overweighted fields.
   *
   * @covers ::calculate
   */
  public function testCalculateWithOverweightedFields(): void {
    $entity = $this->createMockEntity([
      'field_title' => TRUE,
      'field_description' => FALSE,
      'field_author' => FALSE,
    ]);

    $weights = [
      'field_title' => 10,
      'field_description' => 1,
      'field_author' => 1,
    ];

    $score = $this->calculator->calculate($entity, $weights);
    // 10 / (10 + 1 + 1) * 100 = 83.33% = 83% rounded
    $this->assertEquals(83, $score);
  }

  /**
   * Tests calculation with non-existent fields.
   *
   * @covers ::calculate
   */
  public function testCalculateWithNonExistentFields(): void {
    $entity = $this->createMockEntity([
      'field_title' => TRUE,
    ]);

    $weights = [
      'field_title' => 1,
      'field_nonexistent' => 1,
    ];

    $score = $this->calculator->calculate($entity, $weights);
    // Non-existent field is ignored
    // 1 out of 1 = 100%
    $this->assertEquals(100, $score);
  }

  /**
   * Creates a mock entity with specified field states.
   *
   * @param array $field_states
   *   Array of field names to boolean values (TRUE = filled, FALSE = empty).
   *
   * @return \Drupal\Core\Entity\ContentEntityInterface
   *   The mock entity.
   */
  protected function createMockEntity(array $field_states): ContentEntityInterface {
    $entity = $this->createMock(ContentEntityInterface::class);

    $entity->method('hasField')
      ->willReturnCallback(function ($field_name) use ($field_states) {
        return isset($field_states[$field_name]);
      });

    $entity->method('get')
      ->willReturnCallback(function ($field_name) use ($field_states) {
        $field = $this->createMock(FieldItemListInterface::class);
        $field_definition = $this->createMock(FieldDefinitionInterface::class);

        $is_filled = $field_states[$field_name] ?? FALSE;

        $field->method('isEmpty')->willReturn(!$is_filled);
        $field->method('getFieldDefinition')->willReturn($field_definition);
        $field_definition->method('getType')->willReturn('string');

        return $field;
      });

    return $entity;
  }

}
