<?php

namespace Drupal\Tests\content_completeness_score\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;

/**
 * Kernel tests for completeness score calculation and storage.
 *
 * @group content_completeness_score
 */
class CompletenessScoreKernelTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'node',
    'field',
    'text',
    'content_completeness_score',
  ];

  /**
   * The completeness calculator service.
   *
   * @var \Drupal\content_completeness_score\Service\CompletenessCalculator
   */
  protected $calculator;

  /**
   * The completeness storage service.
   *
   * @var \Drupal\content_completeness_score\Service\CompletenessScoreStorage
   */
  protected $storage;

  /**
   * The completeness config manager service.
   *
   * @var \Drupal\content_completeness_score\Service\CompletenessConfigManager
   */
  protected $configManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('node');
    $this->installSchema('node', ['node_access']);
    $this->installSchema('content_completeness_score', ['content_completeness_score']);
    $this->installConfig(['node', 'field']);

    // Create a content type.
    $node_type = NodeType::create([
      'type' => 'article',
      'name' => 'Article',
    ]);
    $node_type->save();

    // Get services.
    $this->calculator = $this->container->get('content_completeness_score.calculator');
    $this->storage = $this->container->get('content_completeness_score.storage');
    $this->configManager = $this->container->get('content_completeness_score.config_manager');
  }

  /**
   * Tests that score is calculated and stored on node save.
   */
  public function testScoreCalculationOnNodeSave(): void {
    // Enable completeness scoring for article content type.
    $this->configManager->saveConfig('article', [
      'enabled' => TRUE,
      'weights' => [
        'title' => 1,
        'body' => 1,
      ],
      'total_weight' => 2,
    ]);

    // Create a node with title only.
    $node = Node::create([
      'type' => 'article',
      'title' => 'Test Article',
    ]);
    $node->save();

    // Check that score was calculated and stored.
    $score = $this->storage->getScoreForRevision('node', $node->getRevisionId());
    $this->assertGreaterThan(0, $score);
    $this->assertLessThanOrEqual(100, $score);

    // The score should be 50% since only title is filled (1 out of 2).
    $this->assertEquals(50, $score);
  }

  /**
   * Tests that score is updated on node update.
   */
  public function testScoreUpdateOnNodeUpdate(): void {
    // Enable completeness scoring.
    $this->configManager->saveConfig('article', [
      'enabled' => TRUE,
      'weights' => [
        'title' => 1,
        'body' => 1,
      ],
      'total_weight' => 2,
    ]);

    // Create a node with title only.
    $node = Node::create([
      'type' => 'article',
      'title' => 'Test Article',
    ]);
    $node->save();

    $initial_score = $this->storage->getScoreForRevision('node', $node->getRevisionId());
    $this->assertEquals(50, $initial_score);

    // Update the node to add body.
    $node->body = [
      'value' => 'Test body content',
      'format' => 'plain_text',
    ];
    $node->save();

    // Check that score was recalculated.
    $updated_score = $this->storage->getScoreForRevision('node', $node->getRevisionId());
    $this->assertEquals(100, $updated_score);
  }

  /**
   * Tests that no score is saved when completeness is disabled.
   */
  public function testNoScoreWhenDisabled(): void {
    // Do not enable completeness scoring.
    $this->configManager->saveConfig('article', [
      'enabled' => FALSE,
      'weights' => [],
      'total_weight' => 0,
    ]);

    // Create a node.
    $node = Node::create([
      'type' => 'article',
      'title' => 'Test Article',
    ]);
    $node->save();

    // Check that no score was saved.
    $score = $this->storage->getScoreForRevision('node', $node->getRevisionId());
    $this->assertEquals(0, $score);
  }

  /**
   * Tests score calculation with different field weights.
   */
  public function testScoreWithDifferentWeights(): void {
    // Enable with different weights.
    $this->configManager->saveConfig('article', [
      'enabled' => TRUE,
      'weights' => [
        'title' => 2,
        'body' => 1,
      ],
      'total_weight' => 3,
    ]);

    // Create a node with title only.
    $node = Node::create([
      'type' => 'article',
      'title' => 'Test Article',
    ]);
    $node->save();

    // Score should be 67% (2 out of 3).
    $score = $this->storage->getScoreForRevision('node', $node->getRevisionId());
    $this->assertEquals(67, $score);
  }

  /**
   * Tests that entity revision is not duplicated.
   */
  public function testNoRevisionDuplication(): void {
    // Enable completeness scoring.
    $this->configManager->saveConfig('article', [
      'enabled' => TRUE,
      'weights' => [
        'title' => 1,
      ],
      'total_weight' => 1,
    ]);

    // Create a node.
    $node = Node::create([
      'type' => 'article',
      'title' => 'Test Article',
    ]);
    $node->save();

    $initial_revision_id = $node->getRevisionId();

    // Update without creating new revision.
    $node->setNewRevision(FALSE);
    $node->title = 'Updated Title';
    $node->save();

    // Revision ID should remain the same.
    $this->assertEquals($initial_revision_id, $node->getRevisionId());
  }

  /**
   * Tests score retrieval methods.
   */
  public function testScoreRetrieval(): void {
    // Enable completeness scoring.
    $this->configManager->saveConfig('article', [
      'enabled' => TRUE,
      'weights' => [
        'title' => 1,
      ],
      'total_weight' => 1,
    ]);

    // Create a node.
    $node = Node::create([
      'type' => 'article',
      'title' => 'Test Article',
    ]);
    $node->save();

    // Test getScoreForRevision.
    $score_by_revision = $this->storage->getScoreForRevision('node', $node->getRevisionId());
    $this->assertEquals(100, $score_by_revision);

    // Test getLatestScore.
    $latest_score = $this->storage->getLatestScore('node', $node->id());
    $this->assertEquals(100, $latest_score);

    // Test scoreExists.
    $exists = $this->storage->scoreExists('node', $node->getRevisionId());
    $this->assertTrue($exists);
  }

  /**
   * Tests score deletion on node deletion.
   */
  public function testScoreDeletionOnNodeDelete(): void {
    // Enable completeness scoring.
    $this->configManager->saveConfig('article', [
      'enabled' => TRUE,
      'weights' => [
        'title' => 1,
      ],
      'total_weight' => 1,
    ]);

    // Create a node.
    $node = Node::create([
      'type' => 'article',
      'title' => 'Test Article',
    ]);
    $node->save();

    $node_id = $node->id();
    $revision_id = $node->getRevisionId();

    // Verify score exists.
    $this->assertTrue($this->storage->scoreExists('node', $revision_id));

    // Delete the node.
    $node->delete();

    // Verify score was deleted.
    $this->assertFalse($this->storage->scoreExists('node', $revision_id));
  }

}
