<?php

namespace Drupal\content_completeness_score\Form;

use Drupal\Component\Utility\Html;
use Drupal\Core\Entity\EntityDisplayRepositoryInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\content_completeness_score\Service\CompletenessConfigManager;
use Drupal\node\NodeTypeInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configuration form for completeness score per bundle.
 */
class CompletenessBundleConfigForm extends ConfigFormBase {

  /**
   * Field names that should be ignored for completeness scoring.
   */
  protected const EXCLUDED_BASE_FIELDS = [
    'nid',
    'uuid',
    'vid',
    'type',
    'langcode',
    'revision_timestamp',
    'revision_uid',
    'revision_log',
    'status',
    'created',
    'changed',
    'promote',
    'sticky',
    'default_langcode',
    'revision_translation_affected',
  ];

  /**
   * The entity display repository.
   *
   * @var \Drupal\Core\Entity\EntityDisplayRepositoryInterface
   */
  protected EntityDisplayRepositoryInterface $entityDisplayRepository;

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected EntityFieldManagerInterface $entityFieldManager;

  /**
   * The completeness config manager.
   *
   * @var \Drupal\content_completeness_score\Service\CompletenessConfigManager
   */
  protected CompletenessConfigManager $configManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->entityFieldManager = $container->get('entity_field.manager');
    $instance->entityDisplayRepository = $container->get('entity_display.repository');
    $instance->configManager = $container->get('content_completeness_score.config_manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    // This will be dynamically determined based on the bundle.
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'content_completeness_score_bundle_config_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, NodeTypeInterface $node_type = NULL) {
    if (!$node_type) {
      return $form;
    }

    $form['#attached']['library'][] = 'content_completeness_score/bundle_config_form';

    $bundle = $node_type->id();
    $config = $this->configManager->getConfig($bundle);

    $form['bundle'] = [
      '#type' => 'value',
      '#value' => $bundle,
    ];

    // Get all fields for this bundle.
    $field_definitions = $this->entityFieldManager->getFieldDefinitions('node', $bundle);

    $form['weights'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['content-completeness-weights'],
      ],
    ];

    $form_display = $this->entityDisplayRepository->getFormDisplay('node', $bundle, 'default');
    $group_definitions = $form_display ? $form_display->getThirdPartySettings('field_group') : [];
    $component_settings = $form_display ? $form_display->getComponents() : [];

    $eligible_fields = $this->prepareFieldMetadata($field_definitions, $component_settings);
    $fields_in_groups = [];

    $group_trees = $this->buildRootGroupTrees($group_definitions, $eligible_fields, $fields_in_groups);
    $has_field_elements = FALSE;
    foreach ($group_trees as $group_tree) {
      $group_element = $this->buildGroupElement($group_tree, $eligible_fields, $config);
      if ($group_element) {
        $form['weights'][$group_tree['name']] = $group_element;
        $has_field_elements = TRUE;
      }
    }

    $ungrouped_fields = array_diff_key($eligible_fields, $fields_in_groups);
    if (!empty($ungrouped_fields)) {
      uasort($ungrouped_fields, static function (array $a, array $b): int {
        $a_position = $a['in_display'] ? 0 : 1;
        $b_position = $b['in_display'] ? 0 : 1;
        return $a_position <=> $b_position
          ?: ($a['weight'] <=> $b['weight'])
            ?: strcmp(mb_strtolower((string) $a['definition']->getLabel()), mb_strtolower((string) $b['definition']->getLabel()));
      });

      foreach ($ungrouped_fields as $field_name => $field_info) {
        $field_element = $this->buildFieldElement($field_name, $field_info['definition'], $config);
        if ($field_element) {
          $field_element['#weight'] = $field_info['weight'];
          $field_element['#attributes']['class'][] = 'content-completeness-field--top-level';
          $form['weights']['field_' . $field_name] = $field_element;
          $has_field_elements = TRUE;
        }
      }
    }

    if (!$has_field_elements) {
      $form['weights']['empty_message'] = [
        '#type' => 'container',
        '#markup' => $this->t('No fields available.'),
        '#attributes' => [
          'class' => ['content-completeness-no-fields'],
        ],
      ];
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $bundle = $form_state->getValue('bundle');
    $config = $this->configManager->getConfig($bundle);
    $enabled = (bool) ($config['enabled'] ?? FALSE);
    $weights = [];
    $field_settings = [];
    $total_weight = 0;

    // Process weights.
    $weight_values = $form_state->getValue('weights');
    if ($weight_values) {
      foreach ($weight_values as $field_name => $field_data) {
        $include = !empty($field_data['include']);
        $weight = isset($field_data['weight']) ? (float) $field_data['weight'] : 0;
        if ($weight < 0) {
          $weight = 0;
        }

        $field_settings[$field_name] = [
          'enabled' => $include,
          'weight' => $weight,
        ];

        if ($include) {
          $weights[$field_name] = $weight;
          if ($weight > 0) {
            $total_weight += $weight;
          }
        }
      }
    }

    // Save configuration.
    $this->configManager->saveConfig($bundle, [
      'enabled' => $enabled,
      'weights' => $weights,
      'field_settings' => $field_settings,
      'total_weight' => $total_weight,
    ]);

    parent::submitForm($form, $form_state);

    $this->messenger()->addStatus($this->t('The completeness score configuration has been saved.'));
  }

  /**
   * Filters and enriches field definitions for display.
   *
   * @param \Drupal\Core\Field\FieldDefinitionInterface[] $field_definitions
   *   Field definitions keyed by field name.
   * @param array $component_settings
   *   Form display component settings keyed by field name.
   *
   * @return array
   *   Filtered metadata keyed by field name.
   */
  protected function prepareFieldMetadata(array $field_definitions, array $component_settings): array {
    $eligible = [];

    foreach ($field_definitions as $field_name => $definition) {
      if (!($definition instanceof FieldDefinitionInterface)) {
        continue;
      }
      if (in_array($field_name, static::EXCLUDED_BASE_FIELDS, TRUE)) {
        continue;
      }

      $eligible[$field_name] = [
        'definition' => $definition,
        'weight' => $component_settings[$field_name]['weight'] ?? 0,
        'in_display' => isset($component_settings[$field_name]),
      ];
    }

    return $eligible;
  }

  /**
   * Builds the root group tree definitions with nested children.
   *
   * @param array $group_definitions
   *   Field group definitions from third party settings.
   * @param array $eligible_fields
   *   Filtered eligible field metadata keyed by field name.
   * @param array $fields_in_groups
   *   Output parameter collecting field names that are part of groups.
   *
   * @return array
   *   Group tree data.
   */
  protected function buildRootGroupTrees(array $group_definitions, array $eligible_fields, array &$fields_in_groups): array {
    $group_trees = [];
    $visited = [];

    foreach ($group_definitions as $group_name => $definition) {
      if (!empty($definition['parent_name'])) {
        continue;
      }

      $group_tree = $this->buildGroupTree($group_name, $group_definitions, $eligible_fields, $fields_in_groups, $visited);
      if ($group_tree) {
        $group_trees[] = $group_tree;
      }
    }

    usort($group_trees, static function (array $a, array $b): int {
      return ($a['weight'] <=> $b['weight'])
        ?: strcmp(mb_strtolower((string) $a['label']), mb_strtolower((string) $b['label']));
    });

    return $group_trees;
  }

  /**
   * Recursively builds a group tree.
   *
   * @param string $group_name
   *   The group machine name.
   * @param array $group_definitions
   *   All group definitions.
   * @param array $eligible_fields
   *   Eligible field metadata keyed by field name.
   * @param array $fields_in_groups
   *   Reference to the collected field names present in groups.
   * @param array $visited
   *   Reference to groups currently in the stack to prevent recursion loops.
   *
   * @return array|null
   *   A group tree array or NULL if the group has no eligible children.
   */
  protected function buildGroupTree(string $group_name, array $group_definitions, array $eligible_fields, array &$fields_in_groups, array &$visited): ?array {
    if (isset($visited[$group_name])) {
      return NULL;
    }

    $visited[$group_name] = TRUE;
    $definition = $group_definitions[$group_name] ?? [];
    $children = [];

    foreach ($definition['children'] ?? [] as $child_name) {
      if (isset($group_definitions[$child_name])) {
        $child_group = $this->buildGroupTree($child_name, $group_definitions, $eligible_fields, $fields_in_groups, $visited);
        if ($child_group) {
          $children[] = $child_group;
        }
      }
      elseif (isset($eligible_fields[$child_name])) {
        $fields_in_groups[$child_name] = TRUE;
        $children[] = [
          'type' => 'field',
          'name' => $child_name,
          'weight' => $eligible_fields[$child_name]['weight'],
        ];
      }
    }

    unset($visited[$group_name]);

    if (!$children) {
      return NULL;
    }

    return [
      'type' => 'group',
      'name' => $group_name,
      'label' => $definition['label'] ?? $group_name,
      'weight' => isset($definition['weight']) ? (int) $definition['weight'] : 0,
      'format_type' => $definition['format_type'] ?? 'details',
      'children' => $children,
    ];
  }

  /**
   * Builds a renderable group element with nested child elements.
   *
   * @param array $group_tree
   *   Group tree definition.
   * @param array $eligible_fields
   *   Eligible field metadata keyed by field name.
   * @param array $config
   *   Saved configuration for the bundle.
   *
   * @return array|null
   *   The render array for the group or NULL if no children rendered.
   */
  protected function buildGroupElement(array $group_tree, array $eligible_fields, array $config): ?array {
    $element = [
      '#type' => 'details',
      '#title' => $group_tree['label'],
      '#open' => TRUE,
      '#weight' => $group_tree['weight'],
      '#attributes' => [
        'class' => [
          'content-completeness-group',
          'content-completeness-group--' . Html::cleanCssIdentifier((string) ($group_tree['format_type'] ?? 'container')),
        ],
      ],
    ];

    $has_children = FALSE;
    foreach ($group_tree['children'] as $child) {
      if (($child['type'] ?? '') === 'group') {
        $child_element = $this->buildGroupElement($child, $eligible_fields, $config);
        if ($child_element) {
          $element[$child['name']] = $child_element;
          $has_children = TRUE;
        }
      }
      elseif (($child['type'] ?? '') === 'field') {
        $field_name = $child['name'];
        if (!isset($eligible_fields[$field_name])) {
          continue;
        }

        $field_element = $this->buildFieldElement($field_name, $eligible_fields[$field_name]['definition'], $config);
        if ($field_element) {
          $field_element['#weight'] = $child['weight'];
          $element['field_' . $field_name] = $field_element;
          $has_children = TRUE;
        }
      }
    }

    return $has_children ? $element : NULL;
  }

  /**
   * Builds the render array for an individual field settings row.
   *
   * @param string $field_name
   *   The field machine name.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The field definition.
   * @param array $config
   *   Saved configuration for the bundle.
   *
   * @return array|null
   *   Render array for the field.
   */
  protected function buildFieldElement(string $field_name, FieldDefinitionInterface $field_definition, array $config): ?array {
    $defaults = $this->getFieldDefaults($field_name, $config);

    $element = [
      '#type' => 'container',
      '#attributes' => [
        'class' => [
          'content-completeness-field',
          $defaults['enabled'] ? 'is-enabled' : 'is-disabled',
        ],
        'data-content-completeness-field' => $field_name,
      ],
    ];

    $element['header'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['content-completeness-field__header'],
      ],
      'label' => [
        '#type' => 'html_tag',
        '#tag' => 'span',
        '#value' => $field_definition->getLabel(),
        '#attributes' => [
          'class' => ['content-completeness-field__label'],
        ],
      ],
      'machine_name' => [
        '#type' => 'html_tag',
        '#tag' => 'span',
        '#value' => Html::escape($field_name),
        '#attributes' => [
          'class' => ['content-completeness-field__machine-name'],
        ],
      ],
    ];

    $element['controls'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['content-completeness-field__controls'],
      ],
    ];

    $element['controls']['include'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Include @label field', ['@label' => $field_definition->getLabel()]),
      '#title_display' => 'invisible',
      '#default_value' => $defaults['enabled'],
      '#attributes' => [
        'class' => ['content-completeness-include'],
      ],
      '#parents' => ['weights', $field_name, 'include'],
    ];

    $element['controls']['weight'] = [
      '#type' => 'number',
      '#title' => $this->t('Weight for @label', ['@label' => $field_definition->getLabel()]),
      '#title_display' => 'invisible',
      '#min' => 0,
      '#max' => 100,
      '#step' => 0.1,
      '#default_value' => $defaults['weight'],
      '#size' => 10,
      '#attributes' => [
        'class' => ['content-completeness-weight'],
      ],
      '#parents' => ['weights', $field_name, 'weight'],
      '#states' => [
        'disabled' => [
          ':input[name="weights[' . $field_name . '][include]"]' => ['checked' => FALSE],
        ],
      ],
    ];

    return $element;
  }

  /**
   * Returns the default enabled state and weight for a field.
   *
   * @param string $field_name
   *   The field machine name.
   * @param array $config
   *   Saved configuration array.
   *
   * @return array
   *   Associative array with keys:
   *   - enabled: (bool) Whether the field is enabled.
   *   - weight: (float) The stored weight.
   */
  protected function getFieldDefaults(string $field_name, array $config): array {
    $field_config = $config['field_settings'][$field_name] ?? NULL;

    if (is_array($field_config)) {
      $enabled = (bool) ($field_config['enabled'] ?? TRUE);
      $weight = isset($field_config['weight']) ? (float) $field_config['weight'] : 1.0;
    }
    else {
      $weight = isset($config['weights'][$field_name]) ? (float) $config['weights'][$field_name] : 1.0;
      $enabled = $weight > 0;
    }

    if ($weight < 0) {
      $weight = 0.0;
    }

    return [
      'enabled' => $enabled,
      'weight' => $weight,
    ];
  }

}
