<?php

namespace Drupal\content_completeness_score\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Link;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Url;
use Drupal\content_completeness_score\Service\CompletenessConfigManager;
use Drupal\content_completeness_score\Service\CompletenessScoreStorage;
use Drupal\node\NodeInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a 'Content Completeness Score WProgress' block.
 *
 * @Block(
 *   id = "content_completeness_score_wprogress",
 *   admin_label = @Translation("Content Completeness Score - WProgress"),
 *   category = @Translation("Content Completeness Score"),
 *   context_definitions = {
 *     "node" = @ContextDefinition("entity:node", label = @Translation("Node"), required = FALSE)
 *   }
 * )
 */
class ContentCompletenessScoreWprogressBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * The completeness score storage service.
   *
   * @var \Drupal\content_completeness_score\Service\CompletenessScoreStorage
   */
  protected CompletenessScoreStorage $storage;

  /**
   * The route match service.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected RouteMatchInterface $routeMatch;

  /**
   * The configuration manager.
   *
   * @var \Drupal\content_completeness_score\Service\CompletenessConfigManager
   */
  protected CompletenessConfigManager $configManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = new static($configuration, $plugin_id, $plugin_definition);
    $instance->storage = $container->get('content_completeness_score.storage');
    $instance->routeMatch = $container->get('current_route_match');
    $instance->configManager = $container->get('content_completeness_score.config_manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    $build = [];

    // Try to get node from context or current route.
    $node = $this->getContextValue('node');
    if (!$node instanceof NodeInterface) {
      $node = $this->routeMatch->getParameter('node');
    }

    if (!$node instanceof NodeInterface) {
      return $build;
    }

    $entity_type = $node->getEntityTypeId();
    $revision_id = $node->getRevisionId();

    if (!$revision_id) {
      return $build;
    }

    $score_exists = $this->storage->scoreExists($entity_type, $revision_id);

    if (!$score_exists) {
      $link = Link::fromTextAndUrl(
        $this->t('Content completeness batch update'),
        Url::fromRoute('content_completeness_score.batch_form')
      );

      return [
        '#type' => 'container',
        '#attributes' => ['class' => ['content-completeness-score--no-data']],
        'message' => [
          '#markup' => $this->t('There is no data to calculate the score yet, please update the content or run the batch update in @link.', [
            '@link' => $link->toString(),
          ]),
        ],
      ];
    }

    $score = $this->storage->getScoreForRevision($entity_type, $revision_id);

    $bundle_config = $this->configManager->getConfig($node->bundle());
    $description = trim((string) ($bundle_config['wprogress_description'] ?? ''));

    $build = [
      '#type' => 'component',
      '#component' => 'content_completeness_score:content-completeness-score-wprogress',
      '#props' => [
        'score' => $score,
        'entity' => $node,
      ],
    ];

    if ($description !== '') {
      $build['#props']['description'] = $description;
    }

    return $build;
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheContexts() {
    return ['url.path', 'route'];
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheTags() {
    $tags = parent::getCacheTags();

    // Add node cache tag if available.
    $node = $this->getContextValue('node');
    if (!$node instanceof NodeInterface) {
      $node = $this->routeMatch->getParameter('node');
    }

    if ($node instanceof NodeInterface) {
      $tags = array_merge($tags, $node->getCacheTags());
      $tags[] = 'config:content_completeness_score.settings.' . $node->bundle();
    }

    return $tags;
  }

}
