/**
 * @file
 * Provides JavaScript functionality.
 */

/* global d3, entityMeshData */
(function (Drupal, once) {
  let contentContainer = null;
  let originalContent = '';
  let showingHtml = false;
  let viewBtn = null;

  /**
   * Attaches behavior to initialize the Entity Mesh graph.
   */
  Drupal.behaviors.content_first = {
    attach(context, settings) {
      once('content-first', '#content-first', context).forEach((element) => {
        element.classList.add('processed');
        element
          .querySelector('#view-button')
          .addEventListener('click', this.viewToggle);
        element
          .querySelector('#copy-button')
          .addEventListener('click', this.copyContent);
        element
          .querySelector('#txt-button')
          .addEventListener('click', this.exportToTxt);

        contentContainer = element.querySelector('#content-first-content');
        originalContent = contentContainer.querySelector('pre').innerHTML;
        viewBtn = element.querySelector('#view-button');
      });
    },
    getContent() {
      const div = document.getElementById('content-first-content');
      if (!div) {
        alert("No element with the ID 'content-first-content' was found.");
        return;
      }
      return div.innerText;
    },
    viewToggle() {
      if (!showingHtml) {
        contentContainer.innerHTML = marked.parse(originalContent.replace(/^---[\s\S]*?---\s*\n*/, ''));
        showingHtml = true;
        viewBtn.value = 'Markdown';
      } else {
        contentContainer.innerHTML = `<pre>${originalContent}</pre>`;
        showingHtml = false;
        viewBtn.value = 'HTML';
      }
    },
    /**
     * Copies the text content to the clipboard.
     */
    copyContent() {
      const content = Drupal.behaviors.content_first.getContent();
      if (content) {
        navigator.clipboard.writeText(content).catch((err) => {
          console.error('Failed to copy text: ', err);
        });
      }
    },
    /**
     * Exports the text content to a text file.
     */
    exportToTxt() {
      const content = Drupal.behaviors.content_first.getContent();
      if (content) {
        const blob = new Blob([content], { type: 'text/plain' });
        const link = document.createElement('a');

        link.href = URL.createObjectURL(blob);
        let fileExtension = showingHtml ? '.txt' : '.md';
        link.download = drupalSettings.content_first.filename + fileExtension;
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
      }
    },
  };
})(Drupal, once);
