<?php

namespace Drupal\content_first\Controller;

use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Symfony\Component\HttpFoundation\ResponseHeaderBag;

/**
 * Downloads rendered node content as a ZIP file.
 */
class ContentFirstDownloadController extends ControllerBase {

  /**
   * The private temp store factory.
   *
   * @var \Drupal\Core\TempStore\PrivateTempStoreFactory
   */
  protected $tempStoreFactory;
  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * Constructor.
   *
   * @param \Drupal\Core\TempStore\PrivateTempStoreFactory $temp_store_factory
   *   The private temp store factory.
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system service.
   */
  public function __construct(PrivateTempStoreFactory $temp_store_factory, FileSystemInterface $file_system) {
    $this->tempStoreFactory = $temp_store_factory;
    $this->fileSystem = $file_system;
  }

  /**
   * Creates an instance of the controller.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The container.
   *
   * @return static
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('tempstore.private'),
      $container->get('file_system')
    );
  }

  /**
   * Downloads rendered node content as a ZIP file.
   */
  public function downloadZip() {
    $store = $this->tempStoreFactory->get('content_first');
    $filename = $store->get('download_zip');
    if (!$filename) {
      throw new NotFoundHttpException();
    }

    $uri = 'temporary://' . $filename;
    $realpath = $this->fileSystem->realpath($uri);
    if (!file_exists($realpath)) {
      throw new NotFoundHttpException();
    }

    $response = new BinaryFileResponse($realpath);
    $response->headers->set('Content-Type', 'application/zip');
    $response->headers->set('X-Content-Type-Options', 'nosniff');
    $response->setContentDisposition(ResponseHeaderBag::DISPOSITION_ATTACHMENT, $filename);
    return $response;
  }

}
