<?php

namespace Drupal\content_first\Controller;

use Drupal\Component\Utility\Html;
use Drupal\content_first\RenderedContent;
use Drupal\node\NodeInterface;

/**
 * Returns Node routes.
 */
class NodeController extends EntityController {

  /**
   * Builds the response.
   */
  public function build(NodeInterface $node) {
    $build = [];

    $rendered_content = $this->contentFirstBuilder->buildContent($node, 'full');

    if (!$rendered_content instanceof RenderedContent) {
      return $build;
    }

    $xpath = $rendered_content->getXpath();

    // Generate headers menu items:
    $headers_items = [];
    // Query all h1 to h6 tags.
    $headers = $xpath->query('//h1 | //h2 | //h3 | //h4 | //h5 | //h6');
    if ($headers) {
      $headers_list = $this->buildHeadersListFromDom($headers);
      $headers_tree = $this->buildHeadersMenuTree($headers_list);
    }

    // Main container:
    $build['container'] = [
      '#type' => 'container',
      '#attributes' => [
        'id' => ['content-first'],
      ],
    ];

    // Content container:
    $build['container']['content'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['content'],
      ],
    ];

    $build['container']['content']['actions'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['actions'],
      ],
    ];

    $build['container']['content']['actions']['view'] = [
      '#type' => 'button',
      '#value' => '🗐 ' . $this->t('HTML'),
      '#attributes' => [
        'id' => ['view-button'],
        'class' => ['action-button'],
      ],
    ];

    $build['container']['content']['actions']['copy'] = [
      '#type' => 'button',
      '#value' => '🗐 ' . $this->t('Copy'),
      '#attributes' => [
        'id' => ['copy-button'],
        'class' => ['action-button'],
      ],
    ];

    $build['container']['content']['actions']['txt'] = [
      '#type' => 'button',
      '#value' => '⤓ ' . $this->t('Download'),
      '#attributes' => [
        'id' => ['txt-button'],
        'class' => ['action-button'],
      ],
    ];

    $build['container']['content']['html'] = [
      '#type' => 'container',
      '#markup' => '<pre>' . $rendered_content->getMarkdown() . '</pre>',
      '#attributes' => [
        'id' => ['content-first-content'],
      ],
    ];

    // Headers container:
    $build['container']['headers'] = [
      '#theme' => 'item_list',
      '#items' => $headers_tree,
      '#list_type' => 'ul',
      '#attributes' => [
        'class' => ['headers-menu'],
      ],
    ];

    $filename = preg_replace('/[^A-Za-z0-9_\-]/', '_', $node->label() . '--' . $node->toUrl()->toString());
    $build['#attached'] = [
      'library' => [
        'content_first/content_first',
      ],
      'drupalSettings' => [
        'content_first' => [
          'label' => $node->label(),
          'url' => $node->toUrl()->toString(),
          'filename' => $filename,
        ],
      ],
    ];

    return $build;
  }
  /**
   * Build a flat list of headers from a DOMNodeList of heading elements.
   * Including new missing levels.
   */
  function buildHeadersListFromDom(\DOMNodeList $nodeList) {
    $headers = [];
    $last_level = 0;

    foreach ($nodeList as $node) {
      if (!$node instanceof \DOMElement) {
        continue;
      }

      // Skip if it doesn't start with 'H'.
      $tagName = strtoupper($node->tagName);
      if (strpos($tagName, 'H') !== 0) {
        continue;
      }

      // Skip or handle non-numeric or invalid headings.
      $lvl = (int) substr($tagName, 1);
      if ($lvl <= 0) {
        continue;
      }

      // Add missing levels.
      for ($i = $last_level + 1; $i < $lvl; $i++) {
        $headers[] = [
          '#markup' => "<span class=\"html-tag missing\">H$i</span> " . $this->t('Missing heading!'),
          '#level' => $i,
        ];
      }

      $tagName = Html::escape($node->tagName);
      $textContent = Html::escape($node->textContent);
      $text = "<span class=\"html-tag\">$tagName</span> $textContent";
      $headers[] = [
        '#markup' => $text,
        '#level' => $lvl,
      ];

      $last_level = $lvl;
    }

    return $headers;
  }

  /**
   * Build a multi-level menu items from a DOMNodeList of heading elements.
   *
   * Each DOMNode is expected to be an element like <h2> or <h3>.
   * We interpret the number in the tagName ("H2" => 2, "H3" => 3, etc.)
   * as the nesting level.
   *
   * The returned structure is an array of items, where each item has:
   *   [
   *     '#markup'  => (string) The text inside the heading,
   *     '#level'   => (int)    The heading level,
   *     'children' => array    Nested sub-items,
   *   ]
   *
  * @param array $headers
  *   A flat list of headers generated by buildHeadersListFromDom(). Each item must
  *   be an associative array containing at least '#markup' (string) and '#level' (int).
  *
   * @return array
   *   Nested array of items with '#markup', '#level', 'children'.
   */
  public function buildHeadersMenuTree(array $headers) {
    // Top-level items in its 'children'.
    $virtualRoot = [
      '#markup'  => '',
      '#level'   => 0,
      'children' => [],
    ];

    // Stack to track current nesting context.
    // Each stack element: [ 'level' => int, 'item' => &theItem ]
    // Start with the virtual root.
    $stack = [
      [
        'level' => 0,
        'item'  => &$virtualRoot,
      ],
    ];

    foreach ($headers as $item) {
      // Climb up the stack while the top of stack has a level >= $lvl.
      while (count($stack) > 0 && $stack[count($stack) - 1]['level'] >= $item['#level']) {
        array_pop($stack);
      }

      // Now the top of stack is a heading whose level < $lvl (or virtual root).
      if (!empty($stack)) {
        $parentItem = &$stack[count($stack) - 1]['item'];

        // Create a new item array.
        $newItem = $item + [
          'children' => [],
        ];

        // Append it to the parent's 'children'.
        $parentItem['children'][] = $newItem;

        // Push this new item onto the stack so deeper headings become its
        // children.
        $newIndex = array_key_last($parentItem['children']);
        $stack[] = [
          'level' => $item['#level'],
          'item'  => &$parentItem['children'][$newIndex],
        ];
      }
    }

    // Return everything under the virtual root (the real items).
    return $virtualRoot['children'];
  }

}
