<?php

namespace Drupal\content_first\Plugin\Action;

use Drupal\Core\Action\ActionBase;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Component\Utility\Crypt;
use Drupal\Component\Transliteration\TransliterationInterface;
use Drupal\node\NodeInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * Downloads rendered node content as a Markdown-only ZIP.
 *
 * @Action(
 *   id = "content_first_zip_download",
 *   label = @Translation("Content First Download Markdown ZIP"),
 *   type = "node",
 * )
 */
class ContentFirstDownloadZip extends ActionBase implements ContainerFactoryPluginInterface {

  /**
   * The tempstore service.
   *
   * @var \Drupal\Core\TempStore\PrivateTempStoreFactory
   */
  protected $tempStoreFactory;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The content builder service.
   *
   * @var \Drupal\content_first\Builder\ContentFirstBuilderInterface
   */
  protected $contentBuilder;

  /**
   * The transliteration service.
   *
   * @var \Drupal\Component\Transliteration\TransliterationInterface
   */
  protected $transliteration;

  /**
   * Constructs a ContentFirstDownloadZip object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\TempStore\PrivateTempStoreFactory $temp_store_factory
   *   The private temp store factory.
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system service.
   * @param \Drupal\content_first\Builder\ContentFirstBuilderInterface $content_builder
   *   The content builder service.
   * @param \Drupal\Component\Transliteration\TransliterationInterface $transliteration
   *   The transliteration service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    PrivateTempStoreFactory $temp_store_factory,
    FileSystemInterface $file_system,
    $content_builder,
    TransliterationInterface $transliteration,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->tempStoreFactory = $temp_store_factory;
    $this->fileSystem = $file_system;
    $this->contentBuilder = $content_builder;
    $this->transliteration = $transliteration;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('tempstore.private'),
      $container->get('file_system'),
      $container->get('content_first.builder'),
      $container->get('transliteration')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function executeMultiple(array $entities) {
    $zip = new \ZipArchive();
    $filename = 'content-first-' . time() . '-' . Crypt::randomBytesBase64(8) . '.zip';
    $zip_path = 'temporary://' . $filename;
    $real_path = $this->fileSystem->realpath($zip_path);

    $zip->open($real_path, \ZipArchive::CREATE);

    foreach ($entities as $node) {
      if ($node instanceof NodeInterface) {
        $rendered = $this->contentBuilder->buildContent($node, 'full');
        $title = $this->transliteration->transliterate($node->label(), $node->language()->getId());
        $zip->addFromString($title . '.md', $rendered->getMarkdown());
      }
    }

    $zip->close();

    $store = $this->tempStoreFactory->get('content_first');
    $store->set('download_zip', $filename);

    \Drupal::messenger()->addStatus($this->t('ZIP file generated. <a href="@url">Click here to download</a>.', [
      '@url' => \Drupal\Core\Url::fromRoute('content_first.download_zip')->toString(),
    ]));
  }

  /**
   * {@inheritdoc}
   */
  public function execute($entity = NULL) {
    if ($entity) {
      $this->executeMultiple([$entity]);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function access($object, ?AccountInterface $account = NULL, $return_as_object = FALSE) {
    return $object->access('view', $account, $return_as_object);
  }

}
