<?php

namespace Drupal\content_first\Builder;

use Drupal\Component\Utility\DeprecationHelper;
use Drupal\content_first\RenderedContent;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Renders plain text from a content entity.
 */
class ContentFirstBuilder implements ContentFirstBuilderInterface {

  use StringTranslationTrait;

  /**
   * Used to render HTML from the content entity.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected RendererInterface $renderer;

  /**
   * Used to build the render array for the content entity.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Used to add metatags as markdown attributes.
   *
   * @var \Symfony\Component\DependencyInjection\ContainerInterface
   */
  protected ContainerInterface $container;

  /**
   * Used to get the allowed metatags for attributes.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * Constructor.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer.
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   Container.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   Config factory service.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    RendererInterface $renderer,
    ContainerInterface $container,
    ConfigFactoryInterface $config_factory,
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->renderer = $renderer;
    $this->container = $container;
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public function buildContent(ContentEntityInterface $entity, string $view_mode) : ?RenderedContent {
    $rendered_content = $this->renderContent($entity, $view_mode);

    if (empty($rendered_content)) {
      return NULL;
    }

    // Generate the content:
    // Limit output to specific tags and attributes:
    $allowed_tags = '<h1><h2><h3><h4><h5><h6>';
    $allowed_tags .= '<a><p><br>';
    $allowed_tags .= '<strong><em><b><i><u>';
    $allowed_tags .= '<ul><li><ol>';
    $allowed_tags .= '<table><tr><td><th><tbody><thead><tfoot>';
    $allowed_tags .= '<blockquote><pre><code>';
    // Media & embedding tags:
    $allowed_tags .= '<img><embed>';
    $allowed_tags .= '<video><audio><iframe><object><canvas><svg>';
    $rendered_content = strip_tags($rendered_content, $allowed_tags);

    // Transform not supported Markdown tags so they are still visible:
    $rendered_content = $this->transformSelfClosingTags($rendered_content, ['embed'], ['alt', 'title']);
    $rendered_content = $this->transformPairedTags($rendered_content, [
      'video',
      'audio',
      'iframe',
      'object',
      'canvas',
      'svg',
    ], ['alt', 'title']);

    $config = $this->configFactory->get('content_first.settings');
    $attributes = $config->get('markdown_attributes') ? $this->extractAttributes($entity) : [];
    $rendered_content = new RenderedContent($rendered_content, $attributes);

    return $rendered_content->isEmpty() ? NULL : $rendered_content;

  }

  /**
   * Renders the content.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The content entity.
   * @param string $view_mode
   *   The view mode.
   *
   * @return string
   *   The rendered content.
   */
  protected function renderContent(ContentEntityInterface $entity, string $view_mode) : string {
    $view_builder = $this->entityTypeManager->getViewBuilder($entity->getEntityTypeId());
    $pre_render = $view_builder->view($entity, $view_mode);
    $render_output = DeprecationHelper::backwardsCompatibleCall(
     currentVersion: \Drupal::VERSION,
     deprecatedVersion: '10.3',
     currentCallable: fn() => $this->renderer->renderInIsolation($pre_render),
     deprecatedCallable: fn() => $this->renderer->renderPlain($pre_render),
    );
    return $render_output;
  }

  /**
   * Transform self-closing tags into bracketed tokens.
   *
   * E.g. <img src="foo" alt="bar" /> -> [IMG src="foo" alt="bar"].
   *
   * @param string $content
   *   The HTML to process.
   * @param string[] $tags
   *   Array of tag names, e.g. ['img','br','hr'].
   * @param string[] $attributes
   *   Array of attributes to capture, e.g. ['src','alt','title'].
   *
   * @return string
   *   The processed HTML content.
   */
  public function transformSelfClosingTags(string $content, array $tags, array $attributes): string {
    if (empty($content)) {
      return '';
    }

    // Match <tag ...> or <tag .../> ignoring content.
    $tagPattern = implode('|', array_map('preg_quote', $tags));
    $pattern = '/<(' . $tagPattern . ')\b([^>]*)\/?>/i';

    $content = preg_replace_callback($pattern, function ($matches) use ($attributes) {
      $tagName    = strtolower($matches[1]);
      $attrString = $matches[2] ?? '';

      $collected = [];
      foreach ($attributes as $attr) {
        // Only matching double quotes attributes:
        $attrPattern = '/\b' . preg_quote($attr, '/') . '="([^"]*)"/i';
        if (preg_match($attrPattern, $attrString, $attrMatch)) {
          $collected[] = $attr . '="' . $attrMatch[1] . '"';
        }
      }

      if ($collected) {
        return '[' . strtoupper($tagName) . ' ' . implode(' ', $collected) . ']';
      }
      else {
        return '[' . strtoupper($tagName) . ']';
      }
    }, $content);

    return (empty($content)) ? '' : $content;
  }

  /**
   * Transform paired tags like <iframe ...>...</iframe> into bracketed tokens.
   *
   * E.g.  <iframe title="Title"></iframe> -> [IFRAME title="Title"]
   *
   * @param string $content
   *   The HTML to process.
   * @param string[] $tags
   *   Array of tag names, e.g. ['iframe','video'].
   * @param string[] $attributes
   *   Array of attributes to capture, e.g. ['src','title'].
   *
   * @return string
   *   The processed HTML content.
   */
  public function transformPairedTags(string $content, array $tags, array $attributes): string {
    if (empty($content)) {
      return '';
    }

    $tagPattern = implode('|', array_map('preg_quote', $tags));
    $pattern = '/<(' . $tagPattern . ')\b([^>]*)>([^>]*)<\/\1>/i';

    $content = preg_replace_callback($pattern, function ($matches) use ($attributes) {
      // $matches[1] = the matched tag name (e.g. "iframe")
      // $matches[2] = the attribute string
      // $matches[3] = inner content between <...> and </...> (ignored)
      // Normalize the tag name in uppercase in the final bracket:
      $tagName = strtoupper($matches[1]);

      $attrString = $matches[2] ?? '';
      $collected = [];
      foreach ($attributes as $attr) {
        // Only matching double‐quoted attributes:
        $attrPattern = '/\b' . preg_quote($attr, '/') . '="([^"]*)"/i';
        if (preg_match($attrPattern, $attrString, $m)) {
          $collected[] = $attr . '="' . $m[1] . '"';
        }
      }

      $result = '[' . $tagName;
      if (!empty($collected)) {
        $result .= ' ' . implode(' ', $collected);
      }
      $result .= ']<br />';

      return trim($result);
    }, $content);

    return (empty($content)) ? '' : $content;
  }

  /**
   * Extracts the attributes form an entity.
   *
   * Extracts the title, url and all its metatags.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   Entity.
   */
  public function extractAttributes(ContentEntityInterface $entity) {
    $attributes = [
      'title' => $entity->label(),
    ];

    if ($entity->hasLinkTemplate('canonical')) {
      $attributes['url'] = $entity->toUrl('canonical', ['absolute' => TRUE])->toString();
    }

    if ($this->container->has('metatag.manager')) {
      $config = $this->configFactory->get('content_first.settings');
      $allowed_tags = $config->get('allowed_metatags') ?? [];
      /** @var \Drupal\metatag\MetatagManagerInterface $manager */
      $manager = $this->container->get('metatag.manager');
      $metatags = $manager->tagsFromEntityWithDefaults($entity);
      $context = [
        'entity' => $entity,
      ];

      $this->container->get('module_handler')->alter('metatags', $metatags, $context);
      $token_service = $this->container->get('token');
      foreach ($metatags as $key => $value) {
        if (!in_array($key, $allowed_tags)) {
          continue;
        }
        $value = $token_service->replace($value, [$entity->getEntityTypeId() => $entity]);
        $attributes['meta-' . $key] = trim(strip_tags(html_entity_decode($value)));
      }

    }

    return $attributes;
  }

}
