<?php

namespace Drupal\content_first\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Markup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Content First settings for this site.
 */
class ContentFirstConfigForm extends ConfigFormBase {

  /**
   * Used to get metatag service, if module is enabled.
   *
   * @var \Symfony\Component\DependencyInjection\ContainerInterface
   */
  protected ContainerInterface $serviceContainer;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->serviceContainer = $container;
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'content_first_config';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['content_first.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('content_first.settings');

    $available_metatags = $this->getAvailableMetatags();

    $form['markdown_attributes'] = [
      '#title' => $this->t('Show markdown attributes'),
      '#description' => $this->t('Markdown attributes are used by some tools like mkdocs to enrich the document metadata. It also helps LLMs to gather context.'),
      '#type' => 'checkbox',
      '#default_value' => $config->get('markdown_attributes') ?? TRUE,
    ];

    if (!empty($available_metatags)) {

      $allowed_metatags = $config->get('allowed_metatags') ?: ['title', 'description', 'abstract', 'keywords'];

      $form['allowed_metatags'] = [
        '#type' => 'checkboxes',
        '#title' => $this->t('Allowed Metatags'),
        '#description' => $this->t('Select the metatags that are allowed to be displayed as entity attributes.'),
        '#options' => $available_metatags,
        '#default_value' => array_combine($allowed_metatags, $allowed_metatags),
        '#required' => TRUE,
      ];
    }
    else {
      $form['metatags_disabled'] = [
        '#type' => 'markup',
        '#markup' => Markup::create('To allow adding metatags as content first markdown attributes, please install the <a href="https://drupal.org/project/metatag" target="_blank" rel="noopener" >Metatag</a> module.'),
      ];
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $allowed_metatags = [];
    $available_metatags = $this->getAvailableMetatags();
    if (!empty($available_metatags)) {
      $allowed_metatags = array_filter($form_state->getValue('allowed_metatags'));
    }

    $this->config('content_first.settings')
      ->set('allowed_metatags', array_values($allowed_metatags))
      ->set('markdown_attributes', $form_state->getValue('markdown_attributes'))
      ->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * Get available metatags options.
   *
   * @return array
   *   Array of available metatags.
   */
  protected function getAvailableMetatags() {
    if ($this->serviceContainer->has('metatag.manager')) {
      /** @var \Drupal\metatag\MetatagManagerInterface $metatag_manager */
      $metatag_manager = $this->serviceContainer->get('metatag.manager');
      return array_map(function ($tag) {
        return $tag['label'];
      }, $metatag_manager->sortedTags());
    }
    else {
      return [];
    }
  }

}
