<?php

namespace Drupal\content_language_no_outbound\Plugin\LanguageNegotiation;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\language\Attribute\LanguageNegotiation;
use Drupal\language\Plugin\LanguageNegotiation\LanguageNegotiationContentEntity;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Route;

/**
 * Class for identifying the content translation language.
 */
#[LanguageNegotiation(
  id: LanguageNegotiationContentEntityNoOutbound::METHOD_ID,
  name: new TranslatableMarkup('Content language (no outbound)'),
  types: [LanguageInterface::TYPE_CONTENT],
  weight: -10,
  description: new TranslatableMarkup("Determines the content language from the request parameter named 'language_content_entity' but sets this parameter only for configured paths."),
  config_route_name: 'content_language_no_outbound.negotiation_content_entity'
)]
class LanguageNegotiationContentEntityNoOutbound extends LanguageNegotiationContentEntity {

  /**
   * The language negotiation method ID.
   */
  const METHOD_ID = 'language-content-entity-no-outbound';

  /**
   * Entity path names from configuration to add query parameter.
   *
   * @var array
   */
  protected ?array $outboundEntityPaths = NULL;

  /**
   * {@inheritdoc}
   */
  protected function meetsContentEntityRoutesCondition(Route $outbound_route, Request $request) {
    if (empty($this->getOutboundEntityPaths())) {
      return FALSE;
    }
    $outbound_path_pattern = $outbound_route->getPath();
    $storage = $this->paths[$request] ?? [];
    if (!isset($storage[$outbound_path_pattern])) {
      $storage[$outbound_path_pattern] = FALSE;

      if (!empty($this->getContentEntityPaths()[$outbound_path_pattern])
        && isset($this->getOutboundEntityPaths()[$this->getContentEntityPaths()[$outbound_path_pattern]])) {

        $storage[$outbound_path_pattern] = TRUE;
      }

      $this->paths[$request] = $storage;
    }

    return $storage[$outbound_path_pattern];
  }

  /**
   * Returns the paths for the link templates of all content entities.
   *
   * @return array
   *   An array of all content entity path names, keyed by the corresponding
   *   link template paths.
   */
  protected function getContentEntityPaths() {
    if (!isset($this->contentEntityPaths)) {
      $this->contentEntityPaths = [];
      $entity_types = $this->entityTypeManager->getDefinitions();
      foreach ($entity_types as $entity_type) {
        if ($entity_type->entityClassImplements(ContentEntityInterface::class)) {
          $entity_paths = array_flip($entity_type->getLinkTemplates());
          $this->contentEntityPaths = array_merge($this->contentEntityPaths, $entity_paths);
        }
      }
    }

    return $this->contentEntityPaths;
  }

  /**
   * Get the configured entity path names where the language parameter is added.
   *
   * @return array
   *   An array of entity path names (like "edit-form" or "delete-form", defined
   *   in the annotations of an entity's class) keyed by themselves.
   */
  protected function getOutboundEntityPaths() {
    if (!isset($this->outboundEntityPaths)) {
      $this->outboundEntityPaths = [];
      $config = $this->config->get('content_language_no_outbound.negotiation');
      $paths = $config->get('content_entity_path_names');
      $paths = preg_split('/\r|\n/', $paths, -1, PREG_SPLIT_NO_EMPTY);
      $paths = array_combine($paths, $paths);
      if ($paths) {
        $this->outboundEntityPaths = $paths;
      }
    }
    return $this->outboundEntityPaths;
  }

}
