<?php

declare(strict_types = 1);

namespace Drupal\content_moderation_tabs\Hook;

use Drupal\content_moderation\ContentModerationState;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\views\Entity\View;

class FormAlter {

  /**
   * Implements hook_form_FORM_ID_alter() for workflow-state-edit-form.
   */
  #[Hook('form_workflow_state_edit_form_alter')]
  public function alterWorkflowStateForm(&$form, FormStateInterface $form_state, $form_id) {
    /** @var \Drupal\workflows\WorkflowInterface $workflow */
    $workflow = $form_state->getFormObject()->getEntity();
    /** @var \Drupal\workflows\StateInterface $state */
    $state = $form_state->get('state');
    $state_id = $state->id();
    // Content Moderation Tabs only works with content moderation states.
    if (! $state instanceof ContentModerationState) {
      return;
    }

    $cmt_settings = self::getContentModerationTabConfig($workflow->id(), $state_id)['cmt_settings'];
    $enabled = $cmt_settings['enabled'] ?? FALSE;
    // @TODO stop lying here and make the default blank if i really autoget these.
    $label = $cmt_settings['label'] ?? $state->label();
    $weight = $cmt_settings['weight'] ?? $state->weight() + 25;
    $views_route = $cmt_settings['views_route'] ?? '';
    $fallback_route = $cmt_settings['fallback_route'] ?? '';

    $form['content_moderation_tabs'] = [
      '#type' => 'fieldset',
      '#tree' => TRUE,
      '#title' => t('Content Moderation Tab settings'),
      '#description' => t('Display a tab for content of the @state state alongside the <strong>Content</strong> » <strong>Overview</strong> sub-tab.', ['@state' => $label]),
    ];

    $form['content_moderation_tabs']['enabled'] = [
      '#type' => 'checkbox',
      '#title' => t('Enable tab'),
      '#default_value' => $enabled,
    ];

    $form['content_moderation_tabs']['label'] = [
      '#type' => 'textfield',
      '#title' => t('Tab title'),
      '#default_value' => $label,
      '#description' => t('Leave blank to use the <strong>State label</strong> set above for the tab title.'),
    ];

    $form['content_moderation_tabs']['weight'] = [
      '#type' => 'weight',
      '#title' => t('Tab weight'),
      '#default_value' => $weight,
      '#description' => t('Overview has weight 0 by default and Moderated content, if kept, has a weight of 10.  Higher numbers place the tab later (to the right in English) in the menu bar.'),
      '#delta' => 50,
    ];

    $form['content_moderation_tabs']['views_route'] = [
      '#type' => 'select',
      '#title' => t('View page'),
      '#options' => self::viewsRouteOptions(),
      '#default_value' => $views_route,
      '#description' => t('Choose the Views view (with page display) to use to display content that has the @state state; this view must already exist', ['@state' => $state_id]),
      '#empty_value' => '_none',
      '#empty_option' => '- None -',
    ];

    $form['content_moderation_tabs']['fallback_route'] = [
      '#type' => 'textfield',
      '#title' => t('Fallback route'),
      '#default_value' => $fallback_route,
      '#description' => t('Leave blank to use the built-in fallback route to display content in this state, enter your own route (must accept a "state" parameter as a string), or type "<none>" to not have any route.  You are probably using Views and really do not care about this one way or the other.'),
    ];

    $form['actions']['submit']['#submit'][] = [FormAlter::class, 'submitWorkflowStateContentModerationTab'];
  }

  /**
   * Submit handler to save Content Moderation Tab settings.
   */
  public static function submitWorkflowStateContentModerationTab(&$form, FormStateInterface $form_state) {
    $values = $form_state->getValue('content_moderation_tabs');
    // Normalize our 'none' value.
    $values['views_route'] = ($values['views_route'] === '_none') ? '' : $values['views_route'];
    $workflow_id = $form_state->getFormObject()->getEntity()->id();
    $state_id = $form_state->get('state')->id();
    $tab_config = self::getContentModerationTabConfig($workflow_id, $state_id);
    $index = $tab_config['index'];
    // We don't use these but hey it gets us an empty array if nothing else.
    $cmt_settings = $tab_config['cmt_settings'];

    $tabs = \Drupal::configFactory()->getEditable('content_moderation_tabs.tabs');
    $workflow_states = $tabs->get('workflow_states') ?? [];
    // If we are re-saving CMT information for a workflow state, $index has the
    // integer value (which may be zero) for updating it in our config sequence
    // and $index has a null value if this is a new tab setting for a state.

    // We do not save anything if there are no routes.
    if (!$values['views_route'] && !$values['fallback_route']) {
      if ($values['enabled']) {
        \Drupal::messenger()->addWarning(t("No route was configured here, so the content moderation tab will not be created or saved."));
      }
      if ($index !== NULL) {
        unset($workflow_states[$index]);
        $tabs->set('workflow_states', $workflow_states);
        $tabs->save();
      }
      return;
    }
    $map = [];
    $map['workflow'] = $workflow_id;
    $map['state'] = $state_id;
    $cmt_settings['enabled'] = $values['enabled'];
    $cmt_settings['label'] = $values['label'];
    $cmt_settings['weight'] = $values['weight'];
    $cmt_settings['views_route'] = $values['views_route'];
    $cmt_settings['fallback_route'] = $values['fallback_route'];
    $map['cmt_settings'] = $cmt_settings;
    if ($index !== NULL) {
      $workflow_states[$index] = $map;
    }
    else {
      $workflow_states[] = $map;
    }
    $tabs->set('workflow_states', $workflow_states);
    $tabs->save();
  }

  /**
   * Load content moderation tab data for the given workflow state. 
   */
  public static function getContentModerationTabConfig(string $workflow, string $state) {
    $tab_config = [];
    $workflow_states = \Drupal::config('content_moderation_tabs.tabs')->get('workflow_states') ?? [];
    // The main drawback to not being in third-party settings for *states*,
    // seemingly not available with workflow as the config entity, or otherwise
    // having a better defined structure than 'sequence' is needing this loop,
    // but we do not care much about a minor inefficiency when saving.
    $index = NULL;
    foreach ($workflow_states as $key => $workflow_state) {
      if ($workflow_state['workflow'] === $workflow && $workflow_state['state'] === $state) {
        $index = $key;
        break;
      }
    }
    $tab_config['index'] = $index;
    $tab_config['cmt_settings'] = ($index !== NULL) ? $workflow_state['cmt_settings'] : [];
    return $tab_config;
  }

  /**
   * Return an options array of routes for all enabled page display views on the site.
   */
  public static function viewsRouteOptions() {
    $options = [];
    $views = View::loadMultiple();
    foreach ($views as $view_id => $view) {
      // Do not list disabled Views.
      if (!$view->status()) {
        // Continue.
      }
      // Get all displays ('getDisplay()' needs to be given a display ID).
      foreach ($view->get('display') as $display_id => $display) {
        if ($display['display_plugin'] === 'page') {
          // The display array does not have an explicit 'enabled' key or such. 
          // @TODO check the enabled/disabled status of displays if possible.
          $options["view.$view_id.$display_id"] = $view->label() . ': ' . $display['display_title'];
        }
      }
    }
    return $options;
  }

}