<?php

namespace Drupal\content_moderation_tabs\Plugin\Derivative;

use Drupal\Component\Plugin\Derivative\DeriverBase;
use Drupal\content_moderation\Entity\ContentModerationStateInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\Discovery\ContainerDeriverInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\workflows\Entity\Workflow;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Routing\Route;
use Symfony\Component\Routing\RouterInterface;

/**
 * Generates moderation-related local tasks.
 */
class DynamicLocalTasks extends DeriverBase implements ContainerDeriverInterface {

  use StringTranslationTrait;

  /**
   * The base plugin ID.
   *
   * @var string
   */
  protected $basePluginId;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The moderation information service.
   *
   * @var \Drupal\content_moderation\ModerationInformationInterface
   */
  protected $moderationInfo;

  /**
   * The router.
   *
   * @var \Symfony\Component\Routing\RouterInterface
   */
  protected $router;

  /**
   * Creates a FieldUiLocalTask object.
   *
   * @param string $base_plugin_id
   *   The base plugin ID.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $string_translation
   *   The translation manager.
   * @param \Symfony\Component\Routing\RouterInterface $router
   *   The router.
   */
  public function __construct($base_plugin_id, EntityTypeManagerInterface $entity_type_manager, TranslationInterface $string_translation, RouterInterface $router) {
    $this->entityTypeManager = $entity_type_manager;
    $this->stringTranslation = $string_translation;
    $this->basePluginId = $base_plugin_id;
    $this->router = $router;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, $base_plugin_id) {
    return new static(
      $base_plugin_id,
      $container->get('entity_type.manager'),
      $container->get('string_translation'),
      $container->get('router')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getDerivativeDefinitions($base_plugin_definition) {
    $this->derivatives = [];

    $workflow_states = \Drupal::config('content_moderation_tabs.tabs')->get('workflow_states') ?? [];

    foreach ($workflow_states as $workflow_state) {
      $cmt = $workflow_state['cmt_settings'];
      if (!$cmt['enabled']) {
        continue;
      }
      // If the Views route is both set and valid, we will use it.
      $route_name = NULL;
      if ($cmt['views_route']) {
        // If this loads, our route name is valid.
        $route = $this->router->getRouteCollection()->get($cmt['views_route']);
        if ($route instanceof Route) {
          $route_name = $cmt['views_route'];
        }
      }
      if (!$route_name && $cmt['fallback_route']) {
        $route = $this->router->getRouteCollection()->get($cmt['fallback_route']);
        if ($route instanceof Route) {
          $route_name = $cmt['fallback_route'];
        }
      }
      if (!$route_name) {
        continue;
      }
      // Consistent ID to use for our derivatives.
      $id = $workflow_state['workflow'] . '.' . $workflow_state['state'];
      $this->derivatives[$id] = [
        'route_name' => $route_name,
        'title' => $cmt['label'],
        'parent_id' => 'system.admin_content',
        'weight' => $cmt['weight'],
      ];
    }

    return $this->derivatives;
  }

}