<?php

namespace Drupal\content_reminders\Controller;

use Drupal\content_reminders\Entity\ReminderEntity;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\content_reminders\ContentReminderInterface;
use Drupal\Core\Link;
use Drupal\node\Entity\Node;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for Content Reminders.
 */
class ContentReminderController extends ControllerBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  final public function __construct(EntityTypeManagerInterface $entity_type_manager) {
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager')
    );
  }

  /**
   * Function to preview a content reminder.
   *
   * @param string $content_reminder
   *   The id of the content reminder.
   *
   * @return array
   *   Render array for the items in the content reminder.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function preview($content_reminder) {
    $entity = $this->entityTypeManager()
      ->getStorage('reminder_entity')
      ->load($content_reminder);

    // If there is no Content Entity check for a Config Entity.
    // @todo Remove this in a future major release.
    if (!$entity) {
      $entity = $this->entityTypeManager()
        ->getStorage('content_reminder')
        ->load($content_reminder);
    }

    // @todo In a future major release only check for ReminderEntity.
    if (!$entity instanceof ContentReminderInterface && !$entity instanceof ReminderEntity) {
      return [];
    }

    // @todo Remove ContentReminderInterface check in a future major release.
    if ($entity instanceof ContentReminderInterface) {
      $items = [
        'Label: ' => $entity->label(),
        'ID:' => $entity->id(),
        'Status' => $entity->status() ? $this->t('Enabled') : $this->t('Disabled'),
        'Node Id:' => $entity->getNodeId(),
        'Emails:' => $entity->getEmails(),
        'Date/Time:' => new DrupalDateTime('@' . $entity->getDateTime()),
        'Message' => $entity->getMessage(),
      ];
    }
    else {
      $items = [
        'Label: ' => $entity->get('label')->value,
        'ID:' => $entity->get('id')->value,
        'Status' => $entity->get('status')->value ? $this->t('Enabled') : $this->t('Disabled'),
        'Node Id:' => $entity->get('nid')->value,
        'Emails:' => $entity->get('emails')->value,
        'Date/Time:' => new DrupalDateTime('@' . $entity->get('date_time')->value),
        'Message' => $entity->get('message')->value,
        'Author' => $this->getUserName($entity->get('uid')->value),
      ];
    }

    return [
      '#theme' => 'item_list',
      '#items' => $items,
    ];
  }

  /**
   * Build out the combined list.
   *
   * @return array
   *   Render array for the list of content reminders.
   *
   * @todo In a future major release move this function's responsibility
   *   to a custom ListBuilder class and remove support for ConfigEntity
   *   content reminders.
   */
  public function build() {
    // Load all the ConfigEntity content reminders.
    $config_entity_storage = $this->entityTypeManager->getStorage('content_reminder');
    $config_entities = $config_entity_storage->loadMultiple();

    // Load all the ContentEntity content reminders.
    $content_entity_storage = $this->entityTypeManager->getStorage('reminder_entity');
    $content_entities = $content_entity_storage->loadMultiple();

    $content_reminders = [];
    foreach ($config_entities as $entity) {
      $node_title = $this->getNodeTitleLink($entity->getNodeId())->getText();
      $user = $this->t('Unknown author');

      $content_reminders[] = [
        'id' => $entity->id(),
        'label' => $entity->label(),
        'status' => $entity->status() ? $this->t('Enabled') : $this->t('Disabled'),
        'nid' => $entity->getNodeId(),
        'node' => $node_title,
        'emails' => $entity->getEmails(),
        'date_time' => new DrupalDateTime('@' . $entity->getDateTime()),
        'message' => $entity->getMessage(),
        'user' => $user,
        'links' => [
          'edit' => $entity->toUrl('edit-form', [$entity->id()]),
          'preview' => $entity->toUrl('preview-page', [$entity->id()]),
          'delete' => $entity->toUrl('delete-form', [$entity->id()]),
        ],
      ];
    }

    foreach ($content_entities as $entity) {
      $node_title = $this->getNodeTitleLink($entity->get('nid')->value)->getText();
      $user = $this->getUserName($entity->get('uid')->value);

      $content_reminders[] = [
        'id' => $entity->get('id')->value,
        'label' => $entity->get('label')->value,
        'status' => $entity->get('status')->value ? $this->t('Enabled') : $this->t('Disabled'),
        'nid' => $entity->get('nid')->value,
        'node' => $node_title,
        'emails' => $entity->get('emails')->value,
        'date_time' => new DrupalDateTime('@' . $entity->get('date_time')->value),
        'message' => $entity->get('message')->value,
        'user' => $user,
        'links' => [
          'edit' => $entity->toUrl('edit-form', [$entity->get('id')->value]),
          'preview' => $entity->toUrl('preview-page', [$entity->get('id')->value]),
          'delete' => $entity->toUrl('delete-form', [$entity->get('id')->value]),
        ],
      ];
    }

    $keys_of_concern = [
      'label',
      'node',
      'status',
      'nid',
      'emails',
      'date_time',
      'message',
      'user',
      'operations',
    ];

    return $this->buildContentRemindersTable($content_reminders, $keys_of_concern);

  }

  /**
   * Retrieve the user name from the user id.
   *
   * @param int $user_id
   *   The user id.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup
   *   The user name of 'Unknown user' if the user cannot be found.
   *
   * @throws \Drupal\Component\Plugin\Exception\Invalid\PluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  private function getUserName($user_id) {
    $username = $this->t('Unknown user');
    if ($user_id) {
      $user_storage = $this->entityTypeManager->getStorage('user');
      $user = $user_storage->load($user_id);
      if ($user) {
        $username = $user->get('name')->value;
      }
    }
    return $username;
  }

  /**
   * Get the link to the node with the node title as the link text.
   *
   * @param int $nid
   *   The node id.
   *
   * @return \Drupal\Core\Link
   *   The link to the node with the node title as the link text.
   *
   * @todo Remove this function in a future major release.
   */
  private function getNodeTitleLink($nid) {
    $node = Node::load($nid);
    $node_title = $node->getTitle();
    $node_title_link = Link::createFromRoute($node_title, 'entity.node.canonical', ['node' => $nid]);

    return $node_title_link;
  }

  /**
   * Build out a render array for the content reminders table.
   *
   * @param array $data
   *   The content reminder data.
   * @param array $keys_of_concern
   *   String array of header text and keys of the data to be displayed.
   *
   * @return array
   *   The render array for the table.
   *
   * @todo Remove this function in a future major release.
   */
  private function buildContentRemindersTable($data, $keys_of_concern) {
    $build['table'] = [
      '#type' => 'table',
      '#header' => $this->getTableHeader(),
      '#rows' => $this->buildTableRows($data, $keys_of_concern),
      '#empty' => $this->t('There are no content reminders yet.'),
    ];

    return $build;
  }

  /**
   * Build out the table header.
   *
   * @return array
   *   The header for the table.
   *
   * @todo Remove this function in a future major release.
   */
  private function getTableHeader() {
    return [
      $this->t('Label'),
      $this->t('Node'),
      $this->t('Status'),
      $this->t('Node Id'),
      $this->t('Emails'),
      $this->t('Date / Time'),
      $this->t('Message'),
      $this->t('User'),
      $this->t('Operations'),
    ];
  }

  /**
   * Build out the table rows.
   *
   * @param array $data
   *   The data to be displayed in the table.
   * @param array $keys
   *   The header text for the table and keys for the data to be displayed.
   *
   * @return array
   *   The rows for the table.
   *
   * @todo Remove this function in a future major release.
   */
  private function buildTableRows($data, $keys) {
    $rows = [];
    foreach ($data as $datum) {
      $row = [];
      foreach ($keys as $key) {
        if ($key === 'operations' || !array_key_exists($key, $datum)) {
          continue;
        }
        else {
          if ($key === 'node') {
            $row[] = $this->getNodeTitleLink($datum['nid']);
          }
          else {
            $row[] = $datum[$key];
          }
        }
      }
      // Build the operations cell as the last cell in the row.
      $row[] = ['data' => $this->buildOperationsCell($datum['links'], $datum['node'])];
      $rows[] = $row;
    }

    return $rows;
  }

  /**
   * Build the operations cell for each row.
   *
   * @param array $link_data
   *   The link data for the operations cell.
   * @param string $node_title
   *   The title of the node for aria-labels.
   *
   * @return array
   *   The render array for the operations cell.
   *
   * @todo Remove this function in a future major release.
   */
  private function buildOperationsCell($link_data, $node_title) {
    $links_array = [];

    foreach ($link_data as $key => $link_value) {
      if ($key === 'edit') {
        $links_array['edit'] = [
          'title' => $this->t('Edit'),
          'url' => $link_value,
          'attributes' => [
            'aria-label' => ['Edit ' . $node_title],
          ],
          'weight' => 10,
        ];
      }
      elseif ($key === 'preview') {
        $links_array['preview'] = [
          'title' => $this->t('Preview'),
          'url' => $link_value,
          'weight' => 20,
        ];
      }
      else {
        $links_array['delete'] = [
          'title' => $this->t('Delete'),
          'url' => $link_value,
          'attributes' => [
            'aria-label' => 'Delete ' . $node_title,
            'class' => ['use-ajax'],
            'data-dialog-type' => 'modal',
            'data-dialog-options' => '{"width":600}',
            'data-once' => 'ajax',
          ],
          'weight' => 30,
        ];
      }
    }

    return [
      '#type' => 'operations',
      '#links' => $links_array,
    ];
  }

}
