<?php

namespace Drupal\content_reminders;

use Drupal\content_reminders\Entity\ReminderEntity;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\InvokeCommand;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Content Reminder Service.
 */
class ContentReminderService {

  use StringTranslationTrait;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|null
   */
  private $entityTypeManager;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  private $currentUser;

  /**
   * {@inheritdoc}
   */
  final public function __construct(EntityTypeManagerInterface $entity_type_manager, AccountProxyInterface $current_user) {
    $this->entityTypeManager = $entity_type_manager;
    $this->currentUser = $current_user;
  }

  /**
   * Function that attaches the reminder form to the node edit form.
   *
   * @param array $form
   *   The form that the Content Reminder form should be attached to.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   Form state object.
   * @param string $form_id
   *   Form ID.
   * @param string|null $entity_id
   *   The entity_id that we are adding the form to.
   */
  public function addReminderToEntityForm(array &$form, FormStateInterface $form_state, $form_id, $entity_id) {

    if ($this->currentUser->isAuthenticated()) {
      $user_id = $this->currentUser->id();
    }
    else {
      $user_id = 0;
    }

    $content_reminder = $this->getContentReminderFor($entity_id);
    if ($content_reminder && !($content_reminder instanceof ReminderEntity)) {
      return;
    }

    $default_emails = '';
    $default_date_time = '';
    $default_message = '';
    if ($content_reminder) {
      $default_emails = $content_reminder->get('emails')->value;
      $default_date_time = new DrupalDateTime('@' . $content_reminder->get('date_time')->value);
      $default_message = $content_reminder->get('message')->value;
    }
    $element['content_reminder'] = [
      '#type' => 'details',
      '#title' => $this->t('Content Reminder'),
      '#collapsed' => FALSE,
      '#collapsible' => TRUE,
      '#tree' => FALSE,
      '#weight' => 20,
      '#group' => 'advanced',
    ];

    $element['content_reminder']['cr_nid'] = [
      '#type' => 'hidden',
      '#value' => $entity_id,
    ];

    $element['content_reminder']['cr_uid'] = [
      '#type' => 'hidden',
      '#value' => $user_id,
    ];

    $element['content_reminder']['cr_emails'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Who to email'),
      '#default_value' => $default_emails,
      '#description' => $this->t('Emails should be separated by a comma.'),
    ];

    $element['content_reminder']['cr_date_time'] = [
      '#type' => 'datetime',
      '#title' => $this->t('Date and Time'),
      '#default_value' => $default_date_time,
      '#description' => $this->t('The date and time that the notification should be sent.'),
    ];

    $element['content_reminder']['cr_message'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Message'),
      '#default_value' => $default_message,
      '#description' => $this->t('An optional note to send with the notification.'),
    ];

    // Currently handled with ajax. Maybe we attach another submit function to
    // the form in order to handle the submission instead? This would require
    // the node edit form to be saved in order to add a content reminder.
    $element['content_reminder']['cr_save'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save'),
      '#submit' => [
        '::submitForm',
        '::save',
      ],
      '#ajax' => [
        'callback' => '_content_reminders_save',
        'wrapper' => 'cr_output',
      ],
    ];

    $element['content_reminder']['output'] = [
      '#prefix' => '<div id="cr_output">',
      '#suffix' => '</div>',
    ];

    $form += $element;
  }

  /**
   * Handles the callback for saving on the node edit form.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   An ajax response.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function handleFormSubmit(array $form, FormStateInterface $form_state) {
    $nid = $form_state->getValue('cr_nid');
    $emails = $form_state->getValue('cr_emails');
    $date_time = $form_state->getValue('cr_date_time');
    $message = $form_state->getValue('cr_message');
    $uid = $form_state->getValue('cr_uid');

    $reminder = $this->getContentReminderFor($nid);

    if (!$reminder) {
      // If no existing reminder, create a new reminder entity.
      $reminder = $this->entityTypeManager
        ->getStorage('reminder_entity')
        ->create();

      $reminder->set('label', $nid . ' content reminder');
      $reminder->set('uid', $uid);
      $reminder->set('nid', $nid);
    }

    // Update the reminder based on its type.
    $reminder->set('status', 1);
    $reminder->set('emails', $emails);
    $reminder->set('date_time', $date_time->getTimestamp());
    $reminder->set('message', $message);
    $reminder->set('uid', $uid);
    $reminder->save();

    // Return an ajax response.
    $response = new AjaxResponse();
    $response->addCommand(new ReplaceCommand('#cr_output', '<div id="cr_output" class="cr-output">Content Reminder Saved (<a href="/admin/structure/content_reminders" target="_blank">View all content reminders</a>)</div>'));
    $response->addCommand(new InvokeCommand('input[name="changed"]', 'val', [time()]));
    return $response;
  }

  /**
   * Get a content reminder for a node.
   *
   * @param string $entity_id
   *   The node id.
   *
   * @return \Drupal\Core\Entity\EntityInterface|null
   *   A content reminder config entity.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  private function getContentReminderFor(string $entity_id) {
    $content_reminder = NULL;

    $query = $this->entityTypeManager
      ->getStorage('reminder_entity')
      ->getQuery();

    $result = $query->condition('nid', $entity_id)
      ->accessCheck(FALSE)
      ->execute();

    if ($result) {
      $key = key($result);
      $content_reminder = $this->entityTypeManager
        ->getStorage('reminder_entity')
        ->load($key);
    }

    return $content_reminder;
  }

}
