<?php

namespace Drupal\content_reminders;

use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityListBuilder;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\user\Entity\User;
use Drupal\Core\Link;

/**
 * List Builder for Content Reminders.
 */
class ContentRemindersListBuilder extends EntityListBuilder {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Constructor for ContentRemindersListBuilder.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type.
   * @param \Drupal\Core\Entity\EntityStorageInterface $storage
   *   The entity storage.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  final public function __construct(EntityTypeInterface $entity_type, EntityStorageInterface $storage, EntityTypeManagerInterface $entity_type_manager) {
    parent::__construct($entity_type, $storage);
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * Create an instance of the list builder.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   Container interface.
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   Entity type interface.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   *
   * @return static
   */
  public static function createInstance(ContainerInterface $container, EntityTypeInterface $entity_type) {
    return new static(
      $entity_type,
      $container->get('entity_type.manager')->getStorage($entity_type->id()),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function buildHeader() {
    $header['label'] = $this->t('Label');
    $header['node'] = $this->t('Node');
    $header['status'] = $this->t('Status');
    $header['nid'] = $this->t('Node Id');
    $header['emails'] = $this->t('Emails');
    $header['date_time'] = $this->t('Date / Time');
    $header['message'] = $this->t('Message');
    $header['uid'] = $this->t('User');

    return $header + parent::buildHeader();
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface $entity) {
    $node = $this->entityTypeManager->getStorage('node')->load($entity->get('nid')->value);
    $node_title = $node->getTitle();
    $node_link = Link::createFromRoute($node_title, 'entity.node.canonical', ['node' => $entity->get('nid')->value]);

    $username = t('Unknown user');
    $user = User::load($entity->get('uid')->value);
    if ($user) {
      $username = $user->getDisplayName();
    }

    $row['label'] = $entity->get('label')->value;
    $row['node'] = $node_link;
    $row['status'] = $entity->get('status')->value ? $this->t('Enabled') : $this->t('Disabled');
    $row['nid'] = $entity->get('nid')->value;
    $row['emails'] = $entity->get('emails')->value;
    $row['date_time'] = new DrupalDateTime('@' . $entity->get('date_time')->value);
    $row['message'] = $entity->get('message')->value;
    $row['uid'] = $username;

    return $row + parent::buildRow($entity);
  }

  /**
   * {@inheritdoc}
   */
  public function getDefaultOperations(EntityInterface $entity) {
    $operations = parent::getDefaultOperations($entity);
    $entity_id = $entity->get('id')->value;
    $operations['preview'] = [
      'title' => $this->t('Preview'),
      'weight' => 20,
      'url' => $this->ensureDestination($entity->toUrl('preview-page', [$entity->get('id')->value])),
    ];
    return $operations;
  }

}
