<?php

namespace Drupal\content_reminders;

use Drupal\content_reminders\Entity\ReminderEntity;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\InvokeCommand;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\Core\Security\TrustedCallbackInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Content Reminder Service.
 */
class ContentReminderService implements TrustedCallbackInterface {

  use StringTranslationTrait;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  private $currentUser;

  /**
   * Constructor for ContentReminderService.
   *
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   */
  final public function __construct(AccountProxyInterface $current_user) {
    $this->currentUser = $current_user;
  }

  /**
   * {@inheritdoc}
   */
  public static function trustedCallbacks() {
    return [
      'clearSubmitMessage',
      'handleFormSubmit',
    ];
  }

  /**
   * Function that attaches the reminder form to the node edit form.
   *
   * @param array $form
   *   The form that the Content Reminder form should be attached to.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   Form state object.
   * @param string $form_id
   *   Form ID.
   * @param string|null $entity_id
   *   The entity_id that we are adding the form to.
   */
  public function addReminderToEntityForm(array &$form, FormStateInterface $form_state, $form_id, $entity_id) {

    if ($this->currentUser->isAuthenticated()) {
      $user_id = $this->currentUser->id();
    }
    else {
      $user_id = 0;
    }

    $content_reminder = $this->getContentReminderFor($entity_id);
    if ($content_reminder && !($content_reminder instanceof ReminderEntity)) {
      return;
    }

    $default_emails = '';
    $default_date_time = '';
    $default_message = '';
    if ($content_reminder) {
      $default_emails = $content_reminder->get('emails')->value;
      $default_date_time = new DrupalDateTime('@' . $content_reminder->get('date_time')->value);
      $default_message = $content_reminder->get('message')->value;
    }

    $collectionLink = Url::fromRoute(
      'entity.reminder_entity.collection'
      );

    $description =
      $this->t(
        'Set up a content reminder for this content item. 
        <a href=":url" target="_blank">View all content reminders</a>.',
        [':url' => $collectionLink->toString()]
      );

    $element['content_reminder'] = [
      '#type' => 'details',
      '#title' => $this->t('Content Reminder'),
      '#collapsed' => FALSE,
      '#collapsible' => TRUE,
      '#tree' => FALSE,
      '#weight' => 20,
      '#group' => 'advanced',
      '#description' => $description,
    ];

    $element['content_reminder']['cr_nid'] = [
      '#type' => 'hidden',
      '#value' => $entity_id,
    ];

    $element['content_reminder']['cr_uid'] = [
      '#type' => 'hidden',
      '#value' => $user_id,
    ];

    $element['content_reminder']['cr_emails'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Who to email'),
      '#default_value' => $default_emails,
      '#description' => $this->t('Emails should be separated by a comma.'),
      '#ajax' => $this->retrieveFocusClearAjaxAttribute(),
    ];

    $element['content_reminder']['cr_date_time'] = [
      '#type' => 'datetime',
      '#title' => $this->t('Date and Time'),
      '#default_value' => $default_date_time,
      '#description' => $this->t('The date and time that the notification should be sent.'),
      '#ajax' => $this->retrieveFocusClearAjaxAttribute(),
    ];

    $element['content_reminder']['cr_message'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Message'),
      '#default_value' => $default_message,
      '#description' => $this->t('An optional note to send with the notification.'),
      '#ajax' => $this->retrieveFocusClearAjaxAttribute(),
    ];

    $element['content_reminder']['output'] = [
      '#prefix' => '<div id="cr_output">',
      '#suffix' => '</div>',
    ];

    // Currently handled with ajax. No need for a submit handler.
    $element['content_reminder']['cr_save'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save Reminder'),
      '#ajax' => $this->retrieveClickSubmitAjaxAttribute(),
    ];

    $form += $element;
  }

  /**
   * Helper to retrieve the ajax attribute for clearing the submit message.
   */
  private function retrieveFocusClearAjaxAttribute() {
    return [
      'callback' => [
        'Drupal\content_reminders\ContentReminderService',
        'clearSubmitMessage',
      ],
      'event' => 'focus',
      'wrapper' => 'cr_output',
      'progress' => [],
    ];
  }

  /**
   * Helper to retrieve the ajax attribute for handling the form submission.
   */
  private function retrieveClickSubmitAjaxAttribute() {
    return [
      'callback' => [
        'Drupal\content_reminders\ContentReminderService',
        'handleFormSubmit',
      ],
      'event' => 'click',
      'wrapper' => 'cr_output',
      'progress' => [],
    ];
  }

  /**
   * Ajax callback to clear the submit message.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   An ajax response.
   */
  public static function clearSubmitMessage(array $form, FormStateInterface $form_state) {
    $response = new AjaxResponse();
    $response->addCommand(new ReplaceCommand('#cr_output', '<div id="cr_output" class="cr-output"></div>'));
    return $response;
  }

  /**
   * Handles the callback for saving on the node edit form.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   An ajax response.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public static function handleFormSubmit(array $form, FormStateInterface $form_state) {

    \Drupal::messenger()->deleteAll();

    // Return an ajax response.
    $response = new AjaxResponse();

    $nid = $form_state->getValue('cr_nid');
    $emails = $form_state->getValue('cr_emails');
    $date_time = $form_state->getValue('cr_date_time');
    $message = $form_state->getValue('cr_message');
    $uid = $form_state->getValue('cr_uid');

    if ($errorMessage = self::checkFormErrors($emails, $date_time, $message)) {
      $response->addCommand(new ReplaceCommand('#cr_output', '<div id="cr_output" class="cr-output error-message">' . $errorMessage . '</div>'));
      return $response;
    }

    /** @var \Drupal\content_reminders\Entity\ReminderEntity $reminder */
    $reminder = self::getContentReminderFor($nid);

    if (!$reminder) {
      /** @var \Drupal\content_reminders\Entity\ReminderEntity $reminder */
      // If no existing reminder, create a new reminder entity.
      $reminder = \Drupal::service('entity_type.manager')
        ->getStorage('reminder_entity')
        ->create();

      $reminder->set('label', $nid . ' content reminder');
      $reminder->set('uid', $uid);
      $reminder->set('nid', $nid);
    }

    // Update the reminder based on its type.
    $reminder->set('status', 1);
    $reminder->set('emails', $emails);
    $reminder->set('date_time', $date_time->getTimestamp());
    $reminder->set('message', $message);
    $reminder->set('uid', $uid);
    $reminder->save();

    $savedMessage = t('Content reminder saved.');

    // Send successful ajax response.
    $response->addCommand(new ReplaceCommand('#cr_output', '<div id="cr_output" class="cr-output">' . $savedMessage . '</div>'));
    $response->addCommand(new InvokeCommand('input[name="changed"]', 'val', [time()]));

    return $response;
  }

  /**
   * Check for form errors.
   *
   * @param string $emails
   *   The emails string.
   * @param \Drupal\Core\Datetime\DrupalDateTime|null|array $date_time
   *   The date time object.
   * @param string $message
   *   The message string.
   *
   * @return string|null
   *   An error message, or null if no errors.
   */
  public static function checkFormErrors($emails, DrupalDateTime|null|array $date_time, $message) {

    /** @var \Drupal\Core\Mail\EmailValidatorInterface $email_validator */
    $email_validator = \Drupal::service('email.validator');

    $email_addresses = array_map('trim', explode(',', $emails));

    if (empty($emails)) {
      return t('Please provide at least one email address.');
    }

    foreach ($email_addresses as $email_address) {

      if (!$email_validator->isValid($email_address)) {
        // The string is not a valid email address.
        return t('The email address %email is not valid.', ['%email' => $email_address]);
      }

    }

    if (
      empty($date_time) ||
      (is_array($date_time) && (empty($date_time['date']) || empty($date_time['time'])))
      ) {
      return t('Please specify a date and time.');
    }

    if (empty(trim($message))) {
      return t('Please provide a message.');
    }

    return NULL;
  }

  /**
   * Get a content reminder for a node.
   *
   * @param string $entity_id
   *   The node id.
   *
   * @return \Drupal\Core\Entity\EntityInterface|null
   *   A content reminder config entity.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  private static function getContentReminderFor(string $entity_id) {
    $content_reminder = NULL;

    $query = \Drupal::service('entity_type.manager')
      ->getStorage('reminder_entity')
      ->getQuery();

    $result = $query->condition('nid', $entity_id)
      ->accessCheck(FALSE)
      ->execute();

    if ($result) {
      $key = key($result);
      $content_reminder = \Drupal::service('entity_type.manager')
        ->getStorage('reminder_entity')
        ->load($key);
    }

    return $content_reminder;
  }

}
