<?php

namespace Drupal\content_reminders;

use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityListBuilder;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\user\Entity\User;

/**
 * List Builder for Content Reminders.
 */
class ContentRemindersListBuilder extends EntityListBuilder {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The current user account.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected AccountProxyInterface $currentUser;

  /**
   * Constructor for ContentRemindersListBuilder.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type.
   * @param \Drupal\Core\Entity\EntityStorageInterface $storage
   *   The entity storage.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   */
  final public function __construct(
    EntityTypeInterface $entity_type,
    EntityStorageInterface $storage,
    EntityTypeManagerInterface $entity_type_manager,
    AccountProxyInterface $current_user,
  ) {
    parent::__construct($entity_type, $storage);
    $this->entityTypeManager = $entity_type_manager;
    $this->currentUser = $current_user;
  }

  /**
   * Create an instance of the list builder.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   Container interface.
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   Entity type interface.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   *
   * @return static
   */
  public static function createInstance(ContainerInterface $container, EntityTypeInterface $entity_type) {
    return new static(
      $entity_type,
      $container->get('entity_type.manager')->getStorage($entity_type->id()),
      $container->get('entity_type.manager'),
      $container->get('current_user')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function buildHeader() {
    $header['nid'] = $this->t('Node Id');
    $header['node'] = $this->t('Node');
    $header['status'] = $this->t('Status');
    $header['emails'] = $this->t('Emails');
    $header['date_time'] = $this->t('Date / Time');
    $header['message'] = $this->t('Message');
    $header['uid'] = $this->t('User');

    return $header + parent::buildHeader();
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface $entity) {
    /** @var \Drupal\content_reminders\Entity\ReminderEntity $entity */
    $node = $this->entityTypeManager->getStorage('node')->load($entity->get('nid')->value);
    $node_title = $node->label();
    $node_link = Link::createFromRoute($node_title, 'entity.node.canonical', ['node' => $entity->get('nid')->value]);

    $username = t('Unknown user');
    $user = User::load($entity->get('uid')->value);
    if ($user) {
      $username = $user->getDisplayName();
    }

    $row['nid'] = $entity->get('nid')->value;
    $row['node'] = $node_link;
    $row['status'] = $entity->get('status')->value ? $this->t('Enabled') : $this->t('Disabled');
    $row['emails'] = $entity->get('emails')->value;
    $row['date_time'] = DrupalDateTime::createFromTimestamp($entity->get('date_time')->value)->format('m/d/Y g:i a');
    $row['message'] = $entity->get('message')->value;
    $row['uid'] = $username;

    return $row + parent::buildRow($entity);
  }

  /**
   * {@inheritdoc}
   */
  public function getDefaultOperations(EntityInterface $entity) {
    /** @var \Drupal\content_reminders\Entity\ReminderEntity $entity */

    $operations = parent::getDefaultOperations($entity);
    $operations['edit']['title'] = $this->t('Edit Reminder');
    $operations['delete']['title'] = $this->t('Delete Reminder');

    if (!$this->currentUser->hasPermission('administer content reminders')) {
      unset($operations['edit']);
    }

    $node_url = Url::fromRoute(
      'entity.node.edit_form',
      ['node' => $entity->get('nid')->value]
    );

    $operations['edit_node'] = [
      'title' => $this->t('Edit Node'),
      'weight' => 20,
      'url' => $this->ensureDestination($node_url),
    ];

    return $operations;
  }

}
