<?php

namespace Drupal\content_reminders\Controller;

use Drupal\content_reminders\Entity\ReminderEntity;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for Content Reminders.
 */
class ContentReminderController extends ControllerBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  final public function __construct(EntityTypeManagerInterface $entity_type_manager) {
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager')
    );
  }

  /**
   * Function to preview a content reminder.
   *
   * @param int $reminder_entity
   *   The id of the content reminder.
   *
   * @return array
   *   Render array for the items in the content reminder.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function preview($reminder_entity) {
    $entity = $this->entityTypeManager()
      ->getStorage('reminder_entity')
      ->load($reminder_entity);

    if (!$entity instanceof ReminderEntity) {
      return [];
    }

    $items = [
      'Label: ' => $entity->get('label')->value,
      'ID:' => $entity->get('id')->value,
      'Status' => $entity->get('status')->value ? $this->t('Enabled') : $this->t('Disabled'),
      'Node Id:' => $entity->get('nid')->value,
      'Emails:' => $entity->get('emails')->value,
      'Date/Time:' => new DrupalDateTime('@' . $entity->get('date_time')->value),
      'Message' => $entity->get('message')->value,
      'Author' => $this->getUserName($entity->get('uid')->value),
    ];

    return [
      '#theme' => 'item_list',
      '#items' => $items,
    ];
  }

  /**
   * Retrieve the user name from the user id.
   *
   * @param int $user_id
   *   The user id.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup
   *   The user name of 'Unknown user' if the user cannot be found.
   *
   * @throws \Drupal\Component\Plugin\Exception\Invalid\PluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  private function getUserName($user_id) {
    $username = $this->t('Unknown user');
    if ($user_id) {
      $user_storage = $this->entityTypeManager->getStorage('user');
      $user = $user_storage->load($user_id);
      if ($user) {
        $username = $user->get('name')->value;
      }
    }
    return $username;
  }

}
