<?php

namespace Drupal\content_reminders\Form;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\content_reminders\Entity\ReminderEntity;
use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Entity\ContentEntityForm;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Content Reminder form.
 */
class ReminderEntityForm extends ContentEntityForm {

  /**
   * The current user service.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected AccountInterface $currentUser;

  /**
   * Constructs a new ReminderEntityForm.
   *
   * @param \Drupal\Core\Entity\EntityRepositoryInterface $entity_repository
   *   The entity repository service.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $entity_type_bundle_info
   *   The entity type bundle service.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   The current user service.
   */
  final public function __construct(EntityRepositoryInterface $entity_repository, EntityTypeBundleInfoInterface $entity_type_bundle_info, TimeInterface $time, AccountInterface $current_user) {
    parent::__construct($entity_repository, $entity_type_bundle_info, $time);
    $this->currentUser = $current_user;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface|ContainerInjectionInterface $container) {
    return new static(
      $container->get('entity.repository'),
      $container->get('entity_type.bundle.info'),
      $container->get('datetime.time'),
      $container->get('current_user')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {

    $form = parent::form($form, $form_state);

    if (!$this->entity instanceof ReminderEntity) {
      return $form;
    }

    // Get the $node object to use it as the $form['nid'] #default_value.
    $node = '';
    $nid = $this->entity->get('nid')->value;
    if ($nid) {
      // Load the node object.
      $node = $this->entityTypeManager->getStorage('node')->load($nid);
    }

    $id = $this->entity->id();
    $status = $this->entity->get('status')->value;
    $label = $this->entity->get('label')->value;
    $emails = $this->entity->get('emails')->value;
    $date_time = new DrupalDateTime('@' . $this->entity->get('date_time')->value);
    $message = $this->entity->get('message')->value;
    $isNew = $this->entity->isNew();
    $load_path = '\Drupal\content_reminders\Entity\ReminderEntity::load';

    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label'),
      '#maxlength' => 255,
      '#default_value' => $label,
      '#description' => $this->t('Label for the content reminder.'),
      '#required' => TRUE,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $id,
      '#machine_name' => [
        'exists' => $load_path,
      ],
      '#disabled' => !$isNew,
    ];

    $form['status'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enabled'),
      '#default_value' => $status,
    ];

    $form['nid'] = [
      '#title' => $this->t('Node'),
      '#description' => $this->t('Start typing to find content.'),
      '#type' => 'entity_autocomplete',
      '#target_type' => 'node',
      '#default_value' => $node,
      '#required' => TRUE,
    ];

    $form['emails'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Who to email'),
      '#default_value' => $emails,
      '#description' => $this->t('Emails should be separated by a comma.'),
      '#required' => TRUE,
    ];

    $form['date_time'] = [
      '#type' => 'datetime',
      '#title' => $this->t('Date and Time'),
      '#default_value' => $date_time,
      '#description' => $this->t('The date and time that the notification should be sent.'),
      '#required' => TRUE,
    ];

    $form['message'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Message'),
      '#default_value' => $message,
      '#description' => $this->t('An optional note to send with the notification.'),
      '#required' => FALSE,
    ];

    if ($this->currentUser->isAuthenticated()) {
      $form['uid'] = [
        '#type' => 'hidden',
        '#value' => $this->currentUser->id(),
      ];
    }

    // Set the form to rebuild. The submitted values are maintained in the
    // form state, and used to build the search results in the form definition.
    $form_state->setRebuild(TRUE);

    return $form;
  }

  /**
   * {@inheritdoc}
   *
   * @throws \Drupal\Core\Entity\EntityMalformedException
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function save(array $form, FormStateInterface $form_state) {
    $reminder = $this->getEntity();
    if (!$reminder instanceof ReminderEntity) {
      return parent::save($form, $form_state);
    }
    $reminder->set('date_time', $form_state->getValue('date_time')->getTimestamp());
    $reminder->set('uid', $form_state->getValue('uid'));
    $result = $reminder->save();

    $message_args = ['%label' => $this->entity->label()];
    $message = $result == SAVED_NEW
      ? $this->t('Created new content reminder %label.', $message_args)
      : $this->t('Updated content reminder %label.', $message_args);
    $this->messenger()->addStatus($message);
    $form_state->setRedirectUrl($this->entity->toUrl('collection'));

    return $result;
  }

}
