<?php

namespace Drupal\content_toc\Plugin\Block;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Block\BlockBase;
use Drupal\Core\Session\AccountInterface;
use Drupal\node\Entity\Node;
use Drupal\Core\Render\Markup;

/**
 * Provides a 'Table of Contents' Block.
 *
 * @Block(
 *   id = "content_toc_block",
 *   admin_label = @Translation("Table of Contents Block")
 * )
 */
class ContentToCBlock extends BlockBase {

  /**
   * {@inheritdoc}
   */
  public function build() {
    $toc = '';

    $config = \Drupal::config('content_toc.settings');
    $enabled_types = $config->get('content_types') ?? [];
    $field_name = $config->get('field_name') ?? 'body';
    $heading_levels = array_filter($config->get('headings') ?? []);

    // Ensure at least one heading level is selected.
    if (empty($heading_levels)) {
      return [];
    }

    // Load the current node.
    $node = \Drupal::routeMatch()->getParameter('node');
    if ($node instanceof Node) {
      // Only proceed if content type is enabled and field exists.
      if (in_array($node->bundle(), $enabled_types) && $node->hasField($field_name)) {
        $body = $node->get($field_name)->value;

        // Parse the body HTML.
        $dom = new \DOMDocument();
        @$dom->loadHTML('<?xml encoding="utf-8" ?>' . $body);

        $xpath = new \DOMXPath($dom);
        $tag_query = implode(' | ', array_map(fn($tag) => '//' . $tag, $heading_levels));
        $headings = $xpath->query($tag_query);

        if ($headings->length > 0) {
          $toc .= '<ul class="toc-list">';
          foreach ($headings as $heading) {
            $text = $heading->textContent;
            $tag = $heading->nodeName;

            // Generate a clean ID.
            $id = strtolower(preg_replace('/[^a-z0-9]+/', '-', $text));
            $heading->setAttribute('id', $id);

            $class = "toc-$tag";
            $toc .= "<li class=\"$class\"><a href=\"#$id\">$text</a></li>";
          }
          $toc .= '</ul>';
        }
      }
    }

    if (empty($toc)) {
      return [];
    }

    return [
      '#markup' => Markup::create($toc),
      '#attached' => [
        'library' => [
          'content_toc/toc',
        ],
      ],
      '#cache' => [
        'max-age' => 0,
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function blockAccess(AccountInterface $account) {
    $config = \Drupal::config('content_toc.settings');
    $enabled_types = $config->get('content_types') ?? [];

    $node = \Drupal::routeMatch()->getParameter('node');
    if ($node instanceof Node && in_array($node->bundle(), $enabled_types)) {
      return AccessResult::allowed();
    }

    return AccessResult::forbidden();
  }

}
