<?php

namespace Drupal\content_translation_access\Plugin\Field\FieldWidget;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\TranslatableInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\Plugin\Field\FieldWidget\LanguageSelectWidget;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'Content Access Translation Language' widget.
 *
 * @FieldWidget(
 *   id = "cta_language_select",
 *   label = @Translation("CTA Language select"),
 *   field_types = {
 *     "language"
 *   }
 * )
 */
final class CTALanguageSelectWidget extends LanguageSelectWidget implements ContainerFactoryPluginInterface {

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  private $languageManager;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  private $currentUser;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  private $entityTypeManager;

  /**
   * {@inheritdoc}
   *
   * @param \Drupal\Core\Field\FieldItemListInterface<\Drupal\Core\Field\FieldItemInterface> $items
   *   The field items.
   * @param int $delta
   *   The field item delta.
   * @param array $element
   *   The form element.
   * @param array $form
   *   The complete form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The form element.
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    $item = $items[$delta];

    $element = parent::formElement($items, $delta, $element, $form, $form_state);
    $languages = $this->languageManager->getLanguages();

    // Check if item exists before calling getEntity()
    if (!$item) {
      return $element;
    }

    /** @var \Drupal\Core\Entity\EntityInterface|null $entity */
    $entity = $item->getEntity();

    // Early return if no entity is available.
    if (!$entity instanceof EntityInterface) {
      return $element;
    }
    $options = [];

    foreach ($languages as $language) {
      $result = AccessResult::neutral();
      if ($entity->isNew()) {
        $result = $this->entityTypeManager
          ->getAccessControlHandler($entity->getEntityTypeId())
          ->createAccess($entity->bundle(), $this->currentUser->getAccount(), ['langcode' => $language->getId()], TRUE);
      }
      else {
        if ($entity instanceof TranslatableInterface) {
          if ($entity->language()->getId() == $language->getId()) {
            $result = AccessResult::allowed();
          }
          else {
            if (!$entity->hasTranslation($language->getId())) {
              $result = $this->entityTypeManager
                ->getAccessControlHandler($entity->getEntityTypeId())
                ->createAccess($entity->bundle(), $this->currentUser->getAccount(), ['langcode' => $language->getId()], TRUE);
            }
            else {
              $translated_entity = $entity->getTranslation($language->getId());
              $result = $this->entityTypeManager
                ->getAccessControlHandler($entity->getEntityTypeId())
                ->access($translated_entity, 'update', $this->currentUser->getAccount(), TRUE);
            }
          }
        }
      }

      if ($result->isAllowed()) {
        $options[$language->getId()] = $language->getName();
      }
    }
    $element['value']['#options'] = $options;
    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function __construct($plugin_id, $plugin_definition, FieldDefinitionInterface $field_definition, array $settings, array $third_party_settings, LanguageManagerInterface $language_manager, AccountProxyInterface $current_user, EntityTypeManagerInterface $entity_type_manager) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $third_party_settings);

    $this->languageManager = $language_manager;
    $this->currentUser = $current_user;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['third_party_settings'],
      $container->get('language_manager'),
      $container->get('current_user'),
      $container->get('entity_type.manager')
    );
  }

}
