<?php

namespace Drupal\content_translation_access\Plugin\Validation\Constraint;

use Drupal\content_translation_access\AccessControlHandlerInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\user\UserStorageInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Validator\Constraint;
use Symfony\Component\Validator\ConstraintValidator;
use Symfony\Component\Validator\Context\ExecutionContextInterface;

/**
 * Validates the CommentName constraint.
 */
final class CreateInLanguageConstraintValidator extends ConstraintValidator implements ContainerInjectionInterface {

  /**
   * Validator 2.5 and upwards compatible execution context.
   *
   * @var \Symfony\Component\Validator\Context\ExecutionContextInterface
   */
  protected ExecutionContextInterface $context;

  /**
   * User storage handler.
   *
   * @var \Drupal\user\UserStorageInterface
   */
  protected $userStorage;

  /**
   * Access control handler.
   *
   * @var \Drupal\content_translation_access\AccessControlHandlerInterface
   */
  protected $accessControlHandler;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a new CommentNameConstraintValidator.
   *
   * @param \Drupal\user\UserStorageInterface $user_storage
   *   The user storage handler.
   * @param \Drupal\content_translation_access\AccessControlHandlerInterface $access_control_handler
   *   Access control handler.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(UserStorageInterface $user_storage, AccessControlHandlerInterface $access_control_handler, EntityTypeManagerInterface $entity_type_manager) {
    $this->userStorage = $user_storage;
    $this->accessControlHandler = $access_control_handler;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
          $container->get('entity_type.manager')
            ->getStorage('user'),
          $container->get('content_translation_access.access_control_handler'),
          $container->get('entity_type.manager')
      );
  }

  /**
   * {@inheritdoc}
   */
  public function validate($value, Constraint $constraint): void {
    if ($value->isNew()) {
      // Get the owner of the entity, or current user if none set.
      $account = $value->getOwner() ?? $this->userStorage->load(1);

      $result = $this->entityTypeManager
        ->getAccessControlHandler($value->getEntityTypeId())
        ->createAccess(
              $value->bundle(), $account, [
                'langcode' => $value->language()
                  ->getId(),
              ], TRUE
            );
      if ($result->isAllowed() === FALSE && $constraint instanceof CreateInLanguageConstraint) {
        $this->context->buildViolation($constraint->invalidLanguage)
          ->addViolation();
      }
    }
  }

}
