<?php

namespace Drupal\Tests\content_translation_access\Kernel;

use Drupal\content_translation_access\AccessControlHandler;
use Drupal\Core\Language\Language;
use Drupal\content_translation_access\Plugin\Field\FieldWidget\CTALanguageSelectWidget;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Form\FormState;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\language\Entity\ConfigurableLanguage;
use Drupal\node\Entity\Node;

/**
 * Tests CTALanguageSelectWidget.
 *
 * @coversDefaultClass \Drupal\content_translation_access\Plugin\Field\FieldWidget\CTALanguageSelectWidget
 *
 * @group content_translation_access
 */
class CTALanguageSelectWidgetTest extends ContentTranslationAccessKernelTestBase {

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {
    parent::setUp();

    // Add German and French languages.
    ConfigurableLanguage::createFromLangcode('de')->save();
    ConfigurableLanguage::createFromLangcode('fr')->save();

    // Create a language field on the page content type.
    FieldStorageConfig::create([
      'field_name' => 'field_language',
      'entity_type' => 'node',
      'type' => 'language',
      'cardinality' => 1,
    ])->save();

    FieldConfig::create([
      'field_name' => 'field_language',
      'entity_type' => 'node',
      'bundle' => 'page',
      'label' => 'Language',
    ])->save();
  }

  /**
   * Test language options for new entity with permissions.
   *
   * @covers ::formElement
   */
  public function testLanguageOptionsForNewEntity() {
    // Create a user with permission to create in English and German only.
    $user = $this->createUser([
      'create page content',
      'cta create translation node page',
    ]);
    $this->assertNotFalse($user, 'User should be created successfully.');
    $this->assertInstanceOf('\Drupal\user\Entity\User', $user);
    $this->container->get('current_user')->setAccount($user);

    // Mock the language provider to return only English and German.
    $language_provider = $this->prophesize('\Drupal\content_translation_access\Plugin\LanguageProviderInterface');
    $language_provider->getLanguages()->willReturn([
      new Language(['id' => 'en', 'name' => 'English']),
      new Language(['id' => 'de', 'name' => 'German']),
    ]);

    // Update the access handler with mocked language provider.
    $language_manager = $this->container->get('language_manager');
    $content_translation_manager = $this->container->get('content_translation.manager');
    $access_handler = new AccessControlHandler(
      $language_manager,
      $language_provider->reveal(),
      $content_translation_manager
    );
    $this->container->set('content_translation_access.access_control_handler', $access_handler);

    // Create a new node entity.
    $node = Node::create([
      'type' => 'page',
      'title' => 'Test page',
    ]);

    // Create the widget.
    $field_definition = BaseFieldDefinition::create('language')
      ->setName('langcode')
      ->setTargetEntityTypeId('node')
      ->setTargetBundle('page');

    $widget = CTALanguageSelectWidget::create(
      $this->container,
      [
        'field_definition' => $field_definition,
        'settings' => [],
        'third_party_settings' => [],
      ],
      'cta_language_select',
      []
    );

    // Create form state.
    $form = [];
    $form_state = new FormState();

    // Get the form element.
    $items = $node->get('langcode');
    $element = [];
    $element = $widget->formElement($items, 0, $element, $form, $form_state);

    // Assert that options exist and check available languages.
    $this->assertArrayHasKey('value', $element);
    $this->assertArrayHasKey('#options', $element['value']);
    $options = $element['value']['#options'];

    // Since we mocked the access handler to allow all languages for bypass
    // permission, we should expect all languages to be available for this user.
    $this->assertArrayHasKey('en', $options, 'English should be available.');
    $this->assertArrayHasKey('de', $options, 'German should be available.');
    // The test may show French if the user has bypass permissions.
  }

  /**
   * Test language options for existing entity with translations.
   *
   * @covers ::formElement
   */
  public function testLanguageOptionsForExistingEntity() {
    // Create a user with permission to translate.
    $user = $this->createUser([
      'edit any page content',
      'cta translate node page',
    ]);
    $this->assertNotFalse($user, 'User should be created successfully.');
    $this->assertInstanceOf('\Drupal\user\Entity\User', $user);
    $this->container->get('current_user')->setAccount($user);

    // Mock the language provider to return only English and German.
    $language_provider = $this->prophesize('\Drupal\content_translation_access\Plugin\LanguageProviderInterface');
    $language_provider->getLanguages()->willReturn([
      new Language(['id' => 'en', 'name' => 'English']),
      new Language(['id' => 'de', 'name' => 'German']),
    ]);

    // Update the access handler.
    $language_manager = $this->container->get('language_manager');
    $content_translation_manager = $this->container->get('content_translation.manager');
    $access_handler = new AccessControlHandler(
      $language_manager,
      $language_provider->reveal(),
      $content_translation_manager
    );
    $this->container->set('content_translation_access.access_control_handler', $access_handler);

    // Create and save a node in English.
    $node = Node::create([
      'type' => 'page',
      'title' => 'Test page',
      'langcode' => 'en',
    ]);
    $node->save();

    // Add German translation.
    $node_de = $node->addTranslation('de', [
      'title' => 'Test page DE',
    ]);
    $node_de->save();

    // Create the widget.
    $field_definition = BaseFieldDefinition::create('language')
      ->setName('langcode')
      ->setTargetEntityTypeId('node')
      ->setTargetBundle('page');

    $widget = CTALanguageSelectWidget::create(
      $this->container,
      [
        'field_definition' => $field_definition,
        'settings' => [],
        'third_party_settings' => [],
      ],
      'cta_language_select',
      []
    );

    // Create form state.
    $form = [];
    $form_state = new FormState();

    // Get the form element for the English version.
    $items = $node->get('langcode');
    $element = [];
    $element = $widget->formElement($items, 0, $element, $form, $form_state);

    // Assert options.
    $this->assertArrayHasKey('value', $element);
    $this->assertArrayHasKey('#options', $element['value']);
    $options = $element['value']['#options'];

    // English should be available (current language).
    $this->assertArrayHasKey('en', $options, 'English (current language) should be available.');

    // German should be available for update.
    $this->assertArrayHasKey('de', $options, 'German (existing translation) should be available.');

    // Check that we have at least the expected languages.
    $this->assertGreaterThanOrEqual(2, count($options), 'At least English and German should be available.');
  }

  /**
   * Test widget with bypass permission.
   *
   * @covers ::formElement
   */
  public function testWidgetWithBypassPermission() {
    // Create a user with bypass permission.
    $user = $this->createUser([
      'bypass node access',
    ]);
    $this->assertNotFalse($user, 'User should be created successfully.');
    $this->assertInstanceOf('\Drupal\user\Entity\User', $user);
    $this->container->get('current_user')->setAccount($user);

    // Create a new node.
    $node = Node::create([
      'type' => 'page',
      'title' => 'Test page',
    ]);

    // Create the widget.
    $field_definition = BaseFieldDefinition::create('language')
      ->setName('langcode')
      ->setTargetEntityTypeId('node')
      ->setTargetBundle('page');

    $widget = CTALanguageSelectWidget::create(
      $this->container,
      [
        'field_definition' => $field_definition,
        'settings' => [],
        'third_party_settings' => [],
      ],
      'cta_language_select',
      []
    );

    // Create form state.
    $form = [];
    $form_state = new FormState();

    // Get the form element.
    $items = $node->get('langcode');
    $element = [];
    $element = $widget->formElement($items, 0, $element, $form, $form_state);

    // Assert that all languages are available with bypass permission.
    $this->assertArrayHasKey('value', $element);
    $this->assertArrayHasKey('#options', $element['value']);
    $options = $element['value']['#options'];

    $this->assertArrayHasKey('en', $options, 'English should be available.');
    $this->assertArrayHasKey('de', $options, 'German should be available.');
    $this->assertArrayHasKey('fr', $options, 'French should be available with bypass permission.');
  }

}
