<?php

namespace Drupal\Tests\content_translation_access\Kernel;

use Drupal\Core\Access\AccessResult;
use Prophecy\Argument;
use Drupal\content_translation_access\Plugin\Validation\Constraint\CreateInLanguageConstraint;
use Drupal\content_translation_access\Plugin\Validation\Constraint\CreateInLanguageConstraintValidator;
use Drupal\language\Entity\ConfigurableLanguage;
use Drupal\node\Entity\Node;

/**
 * Tests CreateInLanguageConstraintValidator.
 *
 * @coversDefaultClass \Drupal\content_translation_access\Plugin\Validation\Constraint\CreateInLanguageConstraintValidator
 *
 * @group content_translation_access
 */
class CreateInLanguageConstraintValidatorTest extends ContentTranslationAccessKernelTestBase {

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {
    parent::setUp();

    // Add German and French languages.
    ConfigurableLanguage::createFromLangcode('de')->save();
    ConfigurableLanguage::createFromLangcode('fr')->save();

    // Add the constraint to the node entity type.
    $entity_type_manager = $this->container->get('entity_type.manager');
    $node_definition = $entity_type_manager->getDefinition('node');
    $constraints = $node_definition->getConstraints();
    $constraints['CreateInLanguage'] = [];
    $node_definition->setConstraints($constraints);
  }

  /**
   * Test validation passes for allowed language.
   *
   * @covers ::validate
   */
  public function testValidationPassesForAllowedLanguage() {
    // Create a user with permission to create in English.
    $user = $this->createUser([
      'cta create translation node page',
    ]);
    $this->assertNotFalse($user, 'User should be created successfully.');
    $this->assertInstanceOf('\Drupal\user\Entity\User', $user);

    // Create a new node in English (allowed language).
    $node = Node::create([
      'type' => 'page',
      'title' => 'Test page',
      'langcode' => 'en',
      'uid' => $user->id(),
    ]);

    // Set the current user.
    $this->container->get('current_user')->setAccount($user);

    // Validate the node.
    $violations = $node->validate();

    // Check that there are no violations for language constraint.
    $language_violations = [];
    foreach ($violations as $violation) {
      if ($violation->getConstraint() instanceof CreateInLanguageConstraint) {
        $language_violations[] = $violation;
      }
    }

    $this->assertCount(0, $language_violations, 'No language constraint violations for allowed language.');
  }

  /**
   * Test validation fails for disallowed language.
   *
   * @covers ::validate
   */
  public function testValidationFailsForDisallowedLanguage() {
    // Create a user without permission to create in German.
    $user = $this->createUser([
      'create page content',
    ]);
    $this->assertNotFalse($user, 'User should be created successfully.');
    $this->assertInstanceOf('\Drupal\user\Entity\User', $user);

    // Mock the node access handler to deny access for German language.
    $node_access_handler = $this->prophesize('\Drupal\Core\Entity\EntityAccessControlHandlerInterface');
    $node_access_handler->createAccess(
      Argument::any(),
      Argument::any(),
      Argument::any(),
      Argument::any()
    )->willReturn(AccessResult::forbidden());

    // Mock the entity type manager to return our mocked access handler.
    $entity_type_manager = $this->prophesize('\Drupal\Core\Entity\EntityTypeManagerInterface');
    $entity_type_manager->getStorage('user')
      ->willReturn($this->container->get('entity_type.manager')->getStorage('user'));
    $entity_type_manager->getAccessControlHandler('node')
      ->willReturn($node_access_handler->reveal());

    // Create a new node in German (disallowed language).
    $node = Node::create([
      'type' => 'page',
      'title' => 'Test page DE',
      'langcode' => 'de',
      'uid' => $user->id(),
    ]);

    // Create the validator with mocked entity type manager.
    $validator = new CreateInLanguageConstraintValidator(
      $this->container->get('entity_type.manager')->getStorage('user'),
      $this->container->get('content_translation_access.access_control_handler'),
      $entity_type_manager->reveal()
    );

    // Create constraint.
    $constraint = new CreateInLanguageConstraint();

    // Create execution context.
    $context = $this->prophesize('\Symfony\Component\Validator\Context\ExecutionContextInterface');
    $violation_builder = $this->prophesize('\Symfony\Component\Validator\Violation\ConstraintViolationBuilderInterface');

    $context->buildViolation($constraint->invalidLanguage)
      ->willReturn($violation_builder->reveal())
      ->shouldBeCalled();

    $violation_builder->addViolation()
      ->shouldBeCalled();

    $validator->initialize($context->reveal());

    // Validate the node.
    $validator->validate($node, $constraint);
  }

  /**
   * Test validation skips existing entities.
   *
   * @covers ::validate
   */
  public function testValidationSkipsExistingEntities() {
    // Create a user.
    $user = $this->createUser([
      'edit any page content',
    ]);
    $this->assertNotFalse($user, 'User should be created successfully.');
    $this->assertInstanceOf('\Drupal\user\Entity\User', $user);

    // Create and save a node.
    $node = Node::create([
      'type' => 'page',
      'title' => 'Test page',
      'langcode' => 'en',
      'uid' => $user->id(),
    ]);
    $node->save();

    // Create the validator.
    $validator = CreateInLanguageConstraintValidator::create($this->container);

    // Create constraint.
    $constraint = new CreateInLanguageConstraint();

    // Create execution context.
    $context = $this->prophesize('\Symfony\Component\Validator\Context\ExecutionContextInterface');

    // buildViolation should NOT be called for existing entities.
    $context->buildViolation(Argument::any())
      ->shouldNotBeCalled();

    $validator->initialize($context->reveal());

    // Validate the existing node (should skip validation).
    $validator->validate($node, $constraint);
  }

  /**
   * Test validation with user having bypass permission.
   *
   * @covers ::validate
   */
  public function testValidationWithBypassPermission() {
    // Create a user with bypass permission.
    $user = $this->createUser([
      'bypass node access',
    ]);
    $this->assertNotFalse($user, 'User should be created successfully.');
    $this->assertInstanceOf('\Drupal\user\Entity\User', $user);

    // Create a new node in any language.
    $node = Node::create([
      'type' => 'page',
      'title' => 'Test page FR',
      'langcode' => 'fr',
      'uid' => $user->id(),
    ]);

    // Set the current user.
    $this->container->get('current_user')->setAccount($user);

    // Validate the node.
    $violations = $node->validate();

    // Check that there are no violations.
    $language_violations = [];
    foreach ($violations as $violation) {
      if ($violation->getConstraint() instanceof CreateInLanguageConstraint) {
        $language_violations[] = $violation;
      }
    }

    $this->assertCount(0, $language_violations, 'No violations for user with bypass permission.');
  }

}
