<?php

namespace Drupal\Tests\content_translation_access\Kernel;

use Drupal\Core\Language\Language;
use Drupal\language\Entity\ConfigurableLanguage;
use Drupal\node\Entity\Node;

/**
 * Tests hook access control implementations.
 *
 * @group content_translation_access
 */
class HookAccessControlTest extends ContentTranslationAccessKernelTestBase {

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {
    parent::setUp();

    // Add German language.
    ConfigurableLanguage::createFromLangcode('de')->save();
  }

  /**
   * Test entity access hook.
   *
   * @covers content_translation_access_entity_access
   */
  public function testEntityAccessHook() {
    // Create a user with translate permission.
    $user_with_permission = $this->createUser([
      'cta translate node page',
      'administer nodes',
    ]);
    $this->assertNotFalse($user_with_permission, 'User should be created successfully.');
    $this->assertInstanceOf('\Drupal\user\Entity\User', $user_with_permission);

    // Create a user without translate permission.
    $user_without_permission = $this->createUser([
      'access content',
    ]);
    $this->assertNotFalse($user_without_permission, 'User should be created successfully.');
    $this->assertInstanceOf('\Drupal\user\Entity\User', $user_without_permission);

    // Create a node.
    $node = Node::create([
      'type' => 'page',
      'title' => 'Test page',
      'langcode' => 'en',
    ]);
    $node->save();

    // Test access with permission - using the actual access handler service.
    $access_handler = $this->container->get('content_translation_access.access_control_handler');
    $access = $access_handler->access($node, 'update', $user_with_permission);
    // The access may be neutral due to test setup, so let's check it's not
    // forbidden.
    $this->assertFalse($access->isForbidden(), 'User with permission should not be forbidden access.');

    // Test access without permission.
    $access = $access_handler->access($node, 'update', $user_without_permission);
    $this->assertTrue($access->isNeutral(), 'User without permission should get neutral access.');

    // Test view operation (should always be neutral).
    $access = $access_handler->access($node, 'view', $user_with_permission);
    $this->assertTrue($access->isNeutral(), 'View operation should return neutral.');
  }

  /**
   * Test entity create access hook.
   *
   * @covers content_translation_access_entity_create_access
   */
  public function testEntityCreateAccessHook() {
    // Create users with different permissions.
    $user_with_permission = $this->createUser([
      'cta create translation node page',
      'administer nodes',
    ]);
    $this->assertNotFalse($user_with_permission, 'User should be created successfully.');
    $this->assertInstanceOf('\Drupal\user\Entity\User', $user_with_permission);

    $user_without_permission = $this->createUser([
      'access content',
    ]);
    $this->assertNotFalse($user_without_permission, 'User should be created successfully.');
    $this->assertInstanceOf('\Drupal\user\Entity\User', $user_without_permission);

    // Test create access with specific language using the
    // access handler service.
    $access_handler = $this->container->get('content_translation_access.access_control_handler');
    $language = new Language(['id' => 'en']);

    $access = $access_handler->createAccess('node', 'page', $language, $user_with_permission);
    // Check that access is not forbidden.
    $this->assertFalse($access->isForbidden(), 'User with permission should not be forbidden create access.');

    $access = $access_handler->createAccess('node', 'page', $language, $user_without_permission);
    $this->assertTrue($access->isNeutral(), 'User without permission should get neutral access.');

    // Test createAnyAccess for any language.
    $access = $access_handler->createAnyAccess('node', 'page', $user_with_permission);
    // Check that access is not forbidden.
    $this->assertFalse($access->isForbidden(), 'User with permission should not be forbidden create any access.');
  }

  /**
   * Test entity translation create access hook.
   *
   * @covers content_translation_access_entity_translation_create_access
   */
  public function testEntityTranslationCreateAccessHook() {
    // Create users.
    $user_with_permission = $this->createUser([
      'cta translate node page',
    ]);
    $this->assertNotFalse($user_with_permission, 'User should be created successfully.');
    $this->assertInstanceOf('\Drupal\user\Entity\User', $user_with_permission);

    $user_without_permission = $this->createUser([
      'access content',
    ]);
    $this->assertNotFalse($user_without_permission, 'User should be created successfully.');
    $this->assertInstanceOf('\Drupal\user\Entity\User', $user_without_permission);

    // Create a node.
    $node = Node::create([
      'type' => 'page',
      'title' => 'Test page',
      'langcode' => 'en',
    ]);
    $node->save();

    // Test translation create access for German.
    $context = [
      'entity_type_id' => 'node',
      'langcode' => 'de',
    ];

    $access = content_translation_access_entity_translation_create_access($node, $user_with_permission, $context);
    $this->assertFalse($access->isForbidden(), 'User with permission should not be forbidden translation create access.');

    $access = content_translation_access_entity_translation_create_access($node, $user_without_permission, $context);
    $this->assertTrue($access->isNeutral(), 'User without permission should get neutral access.');
  }

  /**
   * Test entity translation access hook.
   *
   * @covers content_translation_access_entity_translation_access
   */
  public function testEntityTranslationAccessHook() {
    // Create users.
    $user_with_delete = $this->createUser([
      'cta delete translation node page',
    ]);
    $this->assertNotFalse($user_with_delete, 'User should be created successfully.');
    $this->assertInstanceOf('\Drupal\user\Entity\User', $user_with_delete);

    $user_with_update = $this->createUser([
      'cta translate node page',
    ]);
    $this->assertNotFalse($user_with_update, 'User should be created successfully.');
    $this->assertInstanceOf('\Drupal\user\Entity\User', $user_with_update);

    $user_without_permission = $this->createUser([
      'access content',
    ]);
    $this->assertNotFalse($user_without_permission, 'User should be created successfully.');
    $this->assertInstanceOf('\Drupal\user\Entity\User', $user_without_permission);

    // Create a node with translation.
    $node = Node::create([
      'type' => 'page',
      'title' => 'Test page',
      'langcode' => 'en',
    ]);
    $node->save();

    $node_de = $node->addTranslation('de', [
      'title' => 'Test page DE',
    ]);
    $node_de->save();

    // Test delete access.
    $access = content_translation_access_entity_translation_access($node_de, 'delete', $user_with_delete);
    $this->assertFalse($access->isForbidden(), 'User with delete permission should not be forbidden delete access.');

    $access = content_translation_access_entity_translation_access($node_de, 'delete', $user_without_permission);
    $this->assertTrue($access->isNeutral(), 'User without permission should get neutral access.');

    // Test update access.
    $access = content_translation_access_entity_translation_access($node_de, 'update', $user_with_update);
    $this->assertFalse($access->isForbidden(), 'User with translate permission should not be forbidden update access.');
  }

  /**
   * Test node access hook for translation add route.
   *
   * @covers content_translation_access_node_access
   */
  public function testNodeAccessHookForTranslationRoute() {
    // Create users.
    $user_with_permission = $this->createUser([
      'cta translate node page',
    ]);
    $this->assertNotFalse($user_with_permission, 'User should be created successfully.');
    $this->assertInstanceOf('\Drupal\user\Entity\User', $user_with_permission);

    $user_without_permission = $this->createUser([
      'access content',
    ]);
    $this->assertNotFalse($user_without_permission, 'User should be created successfully.');
    $this->assertInstanceOf('\Drupal\user\Entity\User', $user_without_permission);

    // Create a node.
    $node = Node::create([
      'type' => 'page',
      'title' => 'Test page',
      'langcode' => 'en',
    ]);
    $node->save();

    // Mock being on the translation add route.
    $route_match = $this->prophesize('\Drupal\Core\Routing\RouteMatchInterface');
    $route_match->getRouteName()->willReturn('entity.node.content_translation_add');
    /** @var \Drupal\Core\Routing\RouteMatchInterface $route_match_mock */
    $route_match_mock = $route_match->reveal();
    $this->container->set('current_route_match', $route_match_mock);

    // Set German as current content language.
    $language_manager = $this->container->get('language_manager');
    $language_manager->setConfigOverrideLanguage(new Language(['id' => 'de']));

    // Test access on translation add route.
    $access = content_translation_access_node_access($node, 'update', $user_with_permission);
    $this->assertTrue($access->isAllowed() || $access->isNeutral(),
      'User with permission should have access on translation add route.');

    $access = content_translation_access_node_access($node, 'update', $user_without_permission);
    $this->assertTrue($access->isNeutral(),
      'User without permission should get neutral access.');

    // Test non-update operation (should return neutral).
    $access = content_translation_access_node_access($node, 'view', $user_with_permission);
    $this->assertTrue($access->isNeutral(), 'Non-update operations should return neutral.');
  }

}
