<?php

namespace Drupal\content_translation_access;

use Drupal\content_translation\ContentTranslationManager;
use Drupal\Core\Entity\ContentEntityFormInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Element;
use Drupal\node\NodeTranslationHandler;

/**
 * Class LocalContentTranslationHandler.
 *
 * Extends the "Hide non translatable fields on translation forms" option by
 * hiding non translatable fields on translation forms
 * when the user is missing the permission.
 *
 * @ingroup entity_api
 */
class ContentTranslationAccessHandler extends NodeTranslationHandler {

  /**
   * {@inheritdoc}
   */
  public function entityFormAlter(array &$form, FormStateInterface $form_state, EntityInterface $entity) {
    // Extends NodeTranslationHandler because alter the access to
    // content_translation fields.
    parent::entityFormAlter($form, $form_state, $entity);

    $form_object = $form_state->getFormObject();
    if (!$form_object instanceof ContentEntityFormInterface) {
      return;
    }
    $form_langcode = $form_object->getFormLangcode($form_state);
    $source_langcode = $this->getSourceLangcode($form_state);

    $new_translation = !empty($source_langcode);
    if (!method_exists($entity, 'getTranslationLanguages')) {
      return;
    }
    $translations = $entity->getTranslationLanguages();
    if ($new_translation) {
      // Make sure a new translation does not appear as existing yet.
      unset($translations[$form_langcode]);
    }
    $has_translations = count($translations) > 1;

    if ($new_translation || $has_translations) {
      // Add the form process function.
      $form['#process'][] = [$this, 'hideNonTranslatableFieldsWithPermission'];

    }

  }

  /**
   * Process callback: Hide non translatable fields if missing permission.
   *
   * @see \Drupal\content_translation_access_user\ContentTranslationAccessHandler::entityFormAlter()
   */
  public function hideNonTranslatableFieldsWithPermission($element, FormStateInterface $form_state, $form) {
    static $ignored_types;

    // @todo Find a more reliable way to determine if a form element concerns a
    //   multilingual value.
    if (!isset($ignored_types)) {
      $ignored_types = array_flip(
            ['actions',
              'value',
              'hidden',
              'vertical_tabs',
              'token',
              'details',
            ]
        );
    }

    $form_object = $form_state->getFormObject();

    if ($form_object instanceof ContentEntityFormInterface && $form_object->getEntity() instanceof ContentEntityInterface) {
      $entity = $form_object->getEntity();
    }
    else {
      return $element;
    }

    $settings = [];
    if ($this->manager instanceof ContentTranslationManager && method_exists($this->manager, 'getBundleTranslationSettings')) {
      $settings = $this->manager->getBundleTranslationSettings($entity->getEntityTypeId(), $entity->bundle());
    }
    $hide_translation_fields = (!empty($settings['untranslatable_fields_hide_with_permission']) || ContentTranslationManager::isPendingRevisionSupportEnabled($entity->getEntityTypeId(), $entity->bundle()))
                                && !$this->currentUser->hasPermission('show entity non translatable fields');
    $hide_untranslatable_fields = $hide_translation_fields && !$entity->isDefaultTranslation();
    $translation_form = $form_state->get(['content_translation', 'translation_form']);

    // We use field definitions to identify untranslatable field widgets to be
    // hidden. Fields that are not involved in translation changes checks should
    // not be affected by this logic (the "revision_log" field, for instance).
    $field_definitions = array_diff_key($entity->getFieldDefinitions(), array_flip($this->getFieldsToSkipFromTranslationChangesCheck($entity)));

    foreach (Element::children($element) as $key) {

      if (!isset($element[$key]['#type'])) {
        $this->hideNonTranslatableFieldsWithPermission($element[$key], $form_state, $form);
      }
      else {
        // Ignore non-widget form elements.
        if (isset($ignored_types[$element[$key]['#type']])) {
          continue;
        }
        // Elements are considered to be non multilingual by default.
        if (empty($element[$key]['#multilingual']) && !$translation_form && $hide_untranslatable_fields && isset($field_definitions[$key])) {
          // Hide widgets for untranslatable fields.
          $element[$key]['#access'] = FALSE;
        }
      }
    }

    return $element;
  }

}
